package com.snail.msglet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import tk.ipmsg.IPMessenger;

import java.io.IOException;

import java.util.HashMap;
import java.util.Map;


/**
 * Msgletコンテナ .
 * <pre>
 * このクラスは、全てのMsgletを管理します。
 * このクラスは、IPMessengerを継承しており、メッセージ受信・ユーザ追加
 * ユーザ削除処理を行います。
 * メッセージ受信処理は、現在実行中の Msglet#service(Request, Response)
 * に委譲します。
 * 現在実行中の Msglet が内場合には、msg.xml に記述された、
 * &lt;welcome-msglet&gt が現在実行中の Msglet に設定されます。
 * </pre>
 * @author kagyuu
 */
public final class MsgletContainer extends IPMessenger {
  /**
   * commons logging.
   */
  private static Log logger = LogFactory.getLog(MsgletContainer.class);

  /**
   * 起動されている Msglet の一覧 (key:Msglet名 value:Msglet) .
   */
  private Map<String, Msglet> msgletMap = null;

  /**
   * IPMsg を起動しているユーザの一覧 (key:host value:Userオブジェクト) .
   */
  private Map<String, User> memberMap = new HashMap<String, User>();

  /**
   * デフォルトサーブレット名 .
   */
  private String welcomeServlet;

  /**
   * Creates a new MsgletContainer object.
   *
   * @param userName ユーザ名
   * @param nickName ニックネーム
   * @param group グループ
   * @param debug デバックフラグ
   * @param pWelcomeServlet デフォルトで起動するMsglet名
   * @throws IOException IPMsgが起動できなかったとき
   */
  public MsgletContainer(
    final String userName,
    final String nickName,
    final String group,
    final boolean debug,
    final String pWelcomeServlet) throws IOException {
    super(userName, nickName, group, debug);
    welcomeServlet = pWelcomeServlet;
  }

  /**
   * Msgletの一覧を設定します .
   *
   * @param msgletList Msgletの一覧
   */
  void setMsgletMap(final Map<String, Msglet> msgletList) {
    this.msgletMap = msgletList;
  }

  /**
   * ユーザが追加されたときに、基底クラスのIPMessengerから呼ばれます .
   * <pre>
   * 内部変数の memberaMap にユーザを追加します
   * </pre>
   * @param host ホスト名
   * @param nickName ニックネーム
   * @param group グループ
   * @param addr IPアドレス
   * @param absence 欠席状態フラグ
   */
  @Override
  public void addMember(
    final String host,
    final String nickName,
    final String group,
    final String addr,
    final int absence) {
    memberMap.put(host, new User(host, nickName, group, addr, absence));
  }

  /**
   * ユーザがメールを開いたときに、基底クラスのIPMessengerから呼ばれます .
   * <pre>
   * Msgletでは特に何もしません
   * </pre>
   *
   * @param host ホスト名
   * @param user ユーザ名
   */
  @Override
  public void openMsg(final String host, final String user) {
    logger.info("openMsg/" + host + "/" + user);
  }

  /**
   * ユーザからのメールを受信したときに、基底クラスのIPMessengerから呼ばれます .
   * <pre>
   * 現在実行中の Msglet にメッセージを振り分けます。
   *
   * 現在実行中の Msglet が無い場合には、msg.xml に記載された、welcome-msglet
   * に定義された Msglet が現在実行中の Msglet と見なされます。
   *
   * メッセージが封書形式で送られてきた場合には、開封通知を返信します。
   * </pre>
   * @param host ホスト名
   * @param user ユーザ名
   * @param msg メッセージ
   * @param lock 封書形式で送られてきているか？
   */
  @Override
  public void receiveMsg(
    final String host,
    final String user,
    final String msg,
    final boolean lock) {
    User member = memberMap.get(host);

    if (member == null) {
      logger.error(user + "@" + host + " is lost from my address book");

      try {
        sendMsg(host, "ERROR Lost Session", lock);
      } catch (IOException ignoreEx) {
        ignoreEx = null;
      }

      return;
    }

    // Msgletとセッション中でない場合には welcome-msglet を起動
    String msgletName = member.getCurrentMsgletName();

    if ((msgletName == null) || "".equals(msgletName.trim())) {
      member.setCurrentMsgletName(welcomeServlet);
    }

    // Msgletの起動
    Request req = new Request(member, msg, lock, memberMap, msgletMap);
    Response res = new Response(member, lock);

    Msglet msglet = msgletMap.get(member.getCurrentMsgletName());

    if (msglet != null) {
      msglet.service(req, res);
    } else {
      res.setMsg("Msgletの起動に失敗しました");
    }

    // Msgletの処理結果を送信
    for (User receiver : res.getReceivers()) {
      try {
        sendMsg(receiver.getHost(), res.getMsg(), res.isLock());
      } catch (IOException ex) {
        logger.error(res + "を送信できませんでした", ex);
      }
    }
  }

  /**
   * ユーザが削除されたときに、基底クラスのIPMessengerから呼ばれます .
   * <pre>
   * 内部変数の memberaMap からユーザを削除します
   * </pre>
   * @param host ホスト名
   */
  @Override
  public void removeMember(final String host) {
    memberMap.remove(host);
  }

  /**
   * IPMsgネットワークにログインします .
   *
   * @throws IOException ネットワーク障害発生時
   */
  @Override
  public void login() throws IOException {
    for (String name : msgletMap.keySet()) {
      Msglet msglet = msgletMap.get(name);
      msglet.init();
    }

    super.login();
  }

  /**
   * IPMsgネットワークからログオフします .
   *
   * @throws IOException ネットワーク障害発生時
   */
  @Override
  public void logout() throws IOException {
    for (String name : msgletMap.keySet()) {
      Msglet msglet = msgletMap.get(name);
      msglet.destroy();
    }

    super.logout();
  }
}
