package com.snail.msglet;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.XMLConfiguration;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;


/**
 * msg.xml に従って、Msglet を作成します .
 * @author kagyuu
 */
public final class MsgletFactory {
  /**
   * "msglet"タグ.
   */
  private static final String MSGLET = "msglet";

  /**
   * "msglet-name"タグ.
   */
  private static final String MSGLET_NAME = "msglet-name";

  /**
   * "msglet-class"タグ.
   */
  private static final String MSGLET_CLASS = "msglet-class";

  /**
   * "init-param"タグ.
   */
  private static final String INIT_PARAM = "init-param";

  /**
   * "param-name"タグ.
   */
  private static final String PARAM_NAME = "param-name";

  /**
   * "param-value"タグ.
   */
  private static final String PARAM_VALUE = "param-value";

  /**
   * Creates a new MsgletFactory object.
   */
  private MsgletFactory() {
    super();
  }

  /**
   * msg.xml を元に、Msglet を作ります .
   *
   * @param config msg.xml
   *
   * @return Msgletを格納したMap (Key:Msglet名 , Value:Msglet)
   *
   * @throws InstantiationException Msglet作成失敗
   * @throws IllegalAccessException Msglet作成失敗
   * @throws ClassNotFoundException Msglet作成失敗
   * @throws ConfigurationException Msglet作成失敗
   */
  public static Map<String, Msglet> createMsglets(
    final XMLConfiguration config)
    throws InstantiationException, IllegalAccessException,
      ClassNotFoundException, ConfigurationException {
    Map<String, Msglet> retMap = new HashMap<String, Msglet>();

    // necessary tag
    String pibot = MSGLET + "." + MSGLET_NAME;
    int repetition = getRepetition(config, pibot);

    if (repetition == 1) {
      String name = config.getString(MSGLET + "." + MSGLET_NAME);
      Msglet msglet =
        createMsglet(config.getString(MSGLET + "." + MSGLET_CLASS));

      msglet.setInitParamMap(
        createInitParams(MSGLET + "." + INIT_PARAM, config));

      retMap.put(name, msglet);
    } else if (repetition > 1) {
      for (int cnt = 0; cnt < repetition; cnt++) {
        String name = config.getString(MSGLET + "(" + cnt + ")." + MSGLET_NAME);
        Msglet msglet =
          createMsglet(
              config.getString(MSGLET + "(" + cnt + ")." + MSGLET_CLASS));

        msglet.setInitParamMap(
            createInitParams(MSGLET + "(" + cnt + ")." + INIT_PARAM, config));

        retMap.put(name, msglet);
      }
    }

    return retMap;
  }

  /**
   * あるパス以下�の (param-name, param-value) を Map 形式で返しま�す .
   *
   * @param rootTag (param-name, param-value) があるパス
   * @param config msg.xml
   *
   * @return (param-name, param-value) をマッピングした Map
   */
  private static Map<String, String> createInitParams(
    final String rootTag,
    final XMLConfiguration config) {
    Map<String, String> retMap = new HashMap<String, String>();

    // necessary tag
    final String pibot = rootTag + "." + PARAM_NAME;

    int repetition = getRepetition(config, pibot);

    if (repetition == 1) {
      String paramName = config.getString(rootTag + "." + PARAM_NAME);
      String paramValue = config.getString(rootTag + "." + PARAM_VALUE);

      retMap.put(paramName, paramValue);
    } else if (repetition > 1) {
      for (int cnt = 0; cnt < repetition; cnt++) {
        String paramName =
          config.getString(rootTag + "(" + cnt + ")." + PARAM_NAME);
        String paramValue =
          config.getString(rootTag + "(" + cnt + ")." + PARAM_VALUE);

        retMap.put(paramName, paramValue);
      }
    }

    return retMap;
  }

  /**
   * Msgletをリフレクションによりインスタンス化しま�す .
   *
   * @param className クラス�名
   * @return Msglet
   *
   * @throws InstantiationException インスタンス化に失敗したと�き
   * @throws IllegalAccessException インスタンス化に失敗したと�き
   * @throws ClassNotFoundException インスタンス化に失敗したと�き
   */
  private static Msglet createMsglet(final String className)
    throws InstantiationException, IllegalAccessException,
      ClassNotFoundException {
    return (Msglet) Class.forName(className).newInstance();
  }

  /**
   * 設定ファイルの�なかで、pathで示された属性の繰り返し数を返しま�す .
   * <pre>
   * XMLConfigurationでは�、
   * �複数��回登場するブロック��が何回登場するかを調べるときには�、
   * そ�のブロックの��中で必ず一回��登場する� path ��が何回登場する�か
   * を調べます�。
   * </pre>
   * @param config XML設定ファイル
   * @param path 繰り返し数を調べるpath("a.b.c")
   *
   * @return config のなかでの path の登場数
   */
  private static int getRepetition(
    final XMLConfiguration config,
    final String path) {
    Object obj = config.getProperty(path);

    if (obj == null) {
      return 0;
    } else if (obj instanceof Collection) {
      return ((Collection) obj).size();
    }

    return 1;
  }
}
