﻿using System;
using System.Collections.Generic;
using System.Windows.Forms;
using IrrlichtNetSwig;

namespace ManagedLights
{
 
    static class Program
    {
        static void Main(string[] args)
        {

            E_DRIVER_TYPE driverType = E_DRIVER_TYPE.EDT_DIRECT3D8;
            driverType = IrrlichtNet.driverChoiceConsole();
            if (driverType == E_DRIVER_TYPE.EDT_COUNT)
            {
                return;
            }

            IrrlichtDevice device = IrrlichtNet.createDevice(driverType, new dimension2dui(640, 480), 32,
                                                false, false, false, null);
            if (device == null)
            {
                return;
            }

            float lightRadius = 60.0f; // Enough to reach the far side of each 'zone'

            IVideoDriver driver = device.getVideoDriver();
            ISceneManager smgr = device.getSceneManager();
            IGUIEnvironment guienv = device.getGUIEnvironment();

            IGUISkin skin = guienv.getSkin();
            if (skin != null)
            {
                skin.setColor(EGUI_DEFAULT_COLOR.EGDC_BUTTON_TEXT, new SColor(255, 255, 255, 255));
                IGUIFont font = guienv.getFont("../../media/fontlucida.png");
                if (font != null)
                    skin.setFont(font);
            }

            guienv.addStaticText("1 - No light management", new recti(10, 10, 200, 30));
            guienv.addStaticText("2 - Closest 3 lights", new recti(10, 30, 200, 50));
            guienv.addStaticText("3 - Lights in zone", new recti(10, 50, 200, 70));

            // Add several "zones".  You could use this technique to light individual rooms, for example.
            for (float zoneX = -100.0f; zoneX <= 100.0f; zoneX += 50.0f)
                for (float zoneY = -60.0f; zoneY <= 60.0f; zoneY += 60.0f)
                {
                    // Start with an empty scene node, which we will use to represent a zone.
                    ISceneNode zoneRoot = smgr.addEmptySceneNode();
                    zoneRoot.setPosition(new vector3df(zoneX, zoneY, 0));

                    // Each zone contains a rotating cube
                    IMeshSceneNode node = smgr.addCubeSceneNode(15, zoneRoot);
                    ISceneNodeAnimator rotation = smgr.createRotationAnimator(new vector3df(0.25f, 0.5f, 0.75f));
                    node.addAnimator(rotation);
                    //rotation->drop();

                    // And each cube has three lights attached to it.  The lights are attached to billboards so
                    // that we can see where they are.  The billboards are attached to the cube, so that the
                    // lights are indirect descendents of the same empty scene node as the cube.
                    IBillboardSceneNode billboard = smgr.addBillboardSceneNode(node);
                    billboard.setPosition(new vector3df(0, -14, 30));
                    billboard.setMaterialType(E_MATERIAL_TYPE.EMT_TRANSPARENT_ADD_COLOR);
                    billboard.setMaterialTexture(0, driver.getTexture("../../media/particle.bmp"));
                    billboard.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, false);
                    ILightSceneNode light = smgr.addLightSceneNode(billboard, new vector3df(0, 0, 0), new SColorf(1, 0, 0), lightRadius);

                    billboard = smgr.addBillboardSceneNode(node);
                    billboard.setPosition(new vector3df(-21, -14, -21));
                    billboard.setMaterialType(E_MATERIAL_TYPE.EMT_TRANSPARENT_ADD_COLOR);
                    billboard.setMaterialTexture(0, driver.getTexture("../../media/particle.bmp"));
                    billboard.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, false);
                    light = smgr.addLightSceneNode(billboard, new vector3df(0, 0, 0), new SColorf(0, 1, 0), lightRadius);

                    billboard = smgr.addBillboardSceneNode(node);
                    billboard.setPosition(new vector3df(21, -14, -21));
                    billboard.setMaterialType(E_MATERIAL_TYPE.EMT_TRANSPARENT_ADD_COLOR);
                    billboard.setMaterialTexture(0, driver.getTexture("../../media/particle.bmp"));
                    billboard.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, false);
                    light = smgr.addLightSceneNode(billboard, new vector3df(0, 0, 0), new SColorf(0, 0, 1), lightRadius);

                    // Each cube also has a smaller cube rotating around it, to show that the cubes are being
                    // lit by the lights in their 'zone', not just lights that are their direct children.
                    node = smgr.addCubeSceneNode(5, node);
                    node.setPosition(new vector3df(0, 21, 0));
                }

            smgr.addCameraSceneNode(null, new vector3df(0, 0, -130), new vector3df(0, 0, 0));

            EventReceiver eventReciever = new EventReceiver(smgr);
            //CMyLightManager myLightManager = new CMyLightManager(smgr);
            smgr.setLightManager(null); // This is the default: we won't do light management until told to do it.
            device.setEventReceiver(eventReciever);

            int lastFps = -1;

            while (device.run())
            {
                driver.beginScene(true, true, new SColor(255, 100, 101, 140));
                smgr.drawAll();
                guienv.drawAll();
                driver.endScene();

                int fps = driver.getFPS();
                if (fps != lastFps)
                {
                    lastFps = fps;
                    string str = "Managed Lights [";
                    str += driver.getName();
                    str += "] FPS:";
                    str += fps;
                    device.setWindowCaption(str);
                }
            }

            //myLightManager->drop(); // Drop my implicit reference
            //device->drop();
            return;
        }
    }

   /*
    Normally, you are limited to 8 dynamic lights per scene: this is a hardware limit.  If you
    want to use more dynamic lights in your scene, then you can register an optional light
    manager that allows you to to turn lights on and off at specific point during rendering.
    You are still limited to 8 lights, but the limit is per scene node.

    This is completely optional: if you do not register a light manager, then a default
    distance-based scheme will be used to prioritise hardware lights based on their distance
    from the active camera.

	NO_MANAGEMENT disables the light manager and shows Irrlicht's default light behaviour.
    The 8 lights nearest to the camera will be turned on, and other lights will be turned off.
    In this example, this produces a funky looking but incoherent light display.

	LIGHTS_NEAREST_NODE shows an implementation that turns on a limited number of lights
    per mesh scene node.  If finds the 3 lights that are nearest to the node being rendered,
    and turns them on, turning all other lights off.  This works, but as it operates on every
    light for every node, it does not scale well with many lights.  The flickering you can see
    in this demo is due to the lights swapping their relative positions from the cubes
    (a deliberate demonstration of the limitations of this technique).

	LIGHTS_IN_ZONE shows a technique for turning on lights based on a 'zone'. Each empty scene
    node is considered to be the parent of a zone.  When nodes are rendered, they turn off all
    lights, then find their parent 'zone' and turn on all lights that are inside that zone, i.e.
	are  descendents of it in the scene graph.  This produces true 'local' lighting for each cube
    in this example.  You could use a similar technique to locally light all meshes in (e.g.)
    a room, without the lights spilling out to other rooms.

	This light manager is also an event receiver; this is purely for simplicity in this example,
    it's neither necessary nor recommended for a real application.
    */
    enum LightManagementMode
    {
        NO_MANAGEMENT,
        LIGHTS_NEAREST_NODE,
        LIGHTS_IN_ZONE
    }
    class EventReceiver : IEventReceiver
    {
        CMyLightManager _lightManager;
        ISceneManager SceneManager;
        internal EventReceiver(ISceneManager sceneManager)
        {
            SceneManager = sceneManager;
            _lightManager = new CMyLightManager(sceneManager);
        }

        // The input receiver interface, which just switches light management strategy
        public override bool OnEvent(SEvent ev)
        {
            bool handled = false;

            if (ev.EventType == EEVENT_TYPE.EET_KEY_INPUT_EVENT && ev.KeyInput.PressedDown)
            {
                handled = true;
                switch (ev.KeyInput.Key)
                {
                    case EKEY_CODE.KEY_KEY_1:
                        _lightManager.RequestedMode = LightManagementMode.NO_MANAGEMENT;
                        break;
                    case EKEY_CODE.KEY_KEY_2:
                        _lightManager.RequestedMode = LightManagementMode.LIGHTS_NEAREST_NODE;
                        break;
                    case EKEY_CODE.KEY_KEY_3:
                        _lightManager.RequestedMode = LightManagementMode.LIGHTS_IN_ZONE;
                        break;
                    default:
                        handled = false;
                        break;
                }

                if (LightManagementMode.NO_MANAGEMENT == _lightManager.RequestedMode)
                {
                    SceneManager.setLightManager(null); // Show that it's safe to register the light manager
                }
                else
                {
                    SceneManager.setLightManager(_lightManager);
                }
            }

            return handled;
        }
    }
    class CMyLightManager : ILightManager
    {


	    LightManagementMode Mode;
	    internal LightManagementMode RequestedMode;

	    // These data represent the state information that this light manager
	    // is interested in.
	    ISceneManager SceneManager;
	    arrayLightSceneNode SceneLightList;
	    E_SCENE_NODE_RENDER_PASS CurrentRenderPass;
	    ISceneNode CurrentSceneNode;

        public CMyLightManager(ISceneManager sceneManager)
	    {
            Mode = LightManagementMode.NO_MANAGEMENT;
            RequestedMode = LightManagementMode.NO_MANAGEMENT;
		    SceneManager = sceneManager;
            SceneLightList = null;
            CurrentRenderPass = E_SCENE_NODE_RENDER_PASS.ESNRP_NONE;
            CurrentSceneNode = null;
        }

	    // This is called before the first scene node is rendered.
	    public override void OnPreRender(arrayLightSceneNode lightList)
	    {
		    // Update the mode; changing it here ensures that it's consistent throughout a render
		    Mode = RequestedMode;

		    // Store the light list. I am free to alter this list until the end of OnPostRender().
		    SceneLightList = lightList;
	    }

	    // Called after the last scene node is rendered.
        public override void OnPostRender()
	    {
		    // Since light management might be switched off in the event handler, we'll turn all
		    // lights on to ensure that they are in a consistent state. You wouldn't normally have
		    // to do this when using a light manager, since you'd continue to do light management
		    // yourself.
		    for(uint i = 0; i < SceneLightList.size(); i++)
            {
			    SceneLightList[i].setVisible(true);
            }
	    }

        public override void OnRenderPassPreRender(E_SCENE_NODE_RENDER_PASS renderPass)
	    {
		    // I don't have to do anything here except remember which render pass I am in.
		    CurrentRenderPass = renderPass;
	    }

        public override void OnRenderPassPostRender(E_SCENE_NODE_RENDER_PASS renderPass)
	    {
		    // I only want solid nodes to be lit, so after the solid pass, turn all lights off.
            if (E_SCENE_NODE_RENDER_PASS.ESNRP_SOLID == renderPass)
		    {
			    for(uint i = 0; i < SceneLightList.size(); ++i)
				    SceneLightList[i].setVisible(false);
		    }
	    }

	    // This is called before the specified scene node is rendered
        public override void OnNodePreRender(ISceneNode node)
	    {
		    CurrentSceneNode = node;

		    // This light manager only considers solid objects, but you are free to manipulate
		    // lights during any phase, depending on your requirements.
            if (E_SCENE_NODE_RENDER_PASS.ESNRP_SOLID != CurrentRenderPass)
            {
			    return;
            }

		    // And in fact for this example, I only want to consider lighting for cube scene
		    // nodes.  You will probably want to deal with lighting for (at least) mesh /
		    // animated mesh scene nodes as well.
            if (node.getType() != (int)ESCENE_NODE_TYPE.ESNT_CUBE)
            {
			    return;
            }

            if (LightManagementMode.LIGHTS_NEAREST_NODE == Mode)
		    {
			    // This is a naive implementation that prioritises every light in the scene
			    // by its proximity to the node being rendered.  This produces some flickering
			    // when lights orbit closer to a cube than its 'zone' lights.
			    vector3df nodePosition = node.getAbsolutePosition();

			    // Sort the light list by prioritising them based on their distance from the node
			    // that's about to be rendered.
			    LightDistanceElement[] sortingArray = new LightDistanceElement[SceneLightList.size()];

			    for(uint i = 0; i < SceneLightList.size(); ++i)
			    {
				    ILightSceneNode lightNode = SceneLightList[i];
				    float distance = lightNode.getAbsolutePosition().getDistanceFromSQ(nodePosition);
				    sortingArray[i] = new LightDistanceElement(lightNode, distance);
			    }

                Array.Sort(sortingArray);

			    // The list is now sorted by proximity to the node.
			    // Turn on the three nearest lights, and turn the others off.
			    for(uint i = 0; i < sortingArray.Length; ++i)
                {
				    sortingArray[i].node.setVisible(i < 3);
                }

		    }
            else if (LightManagementMode.LIGHTS_IN_ZONE == Mode)
		    {
			    // Empty scene nodes are used to represent 'zones'.  For each solid mesh that
			    // is being rendered, turn off all lights, then find its 'zone' parent, and turn
			    // on all lights that are found under that node in the scene graph.
			    // This is a general purpose algorithm that doesn't use any special
			    // knowledge of how this particular scene graph is organised.
			    for(uint i = 0; i < SceneLightList.size(); ++i)
			    {
				    ILightSceneNode lightNode = SceneLightList[i];
				    SLight lightData = lightNode.getLightData();

				    if(E_LIGHT_TYPE.ELT_DIRECTIONAL != lightData.Type)
                    {
					    lightNode.setVisible(false);
                    }
			    }

			    ISceneNode parentZone = findZone(node);
			    if(parentZone != null)
                {
				    turnOnZoneLights(parentZone);
                }
		    }
	    }

	    // Called after the specified scene node is rendered
        public override void OnNodePostRender(ISceneNode node)
	    {
		    // I don't need to do any light management after individual node rendering.
	    }



	    // Find the empty scene node that is the parent of the specified node
	    private ISceneNode findZone(ISceneNode node)
	    {
		    if(node == null)
            {
			    return null;
            }

		    if(node.getType() == (int)ESCENE_NODE_TYPE.ESNT_EMPTY)
            {
			    return node;
            }

		    return findZone(node.getParent());
	    }

	    // Turn on all lights that are children (directly or indirectly) of the
	    // specified scene node.
	    private void turnOnZoneLights(ISceneNode node)
	    {
		    listSceneNode children = node.getChildren();
            foreach (ISceneNode child in children)
            {
                if (child.getType() == (int)ESCENE_NODE_TYPE.ESNT_LIGHT)
                {
                    ILightSceneNode.cast(child).setVisible(true);
                }
                else
                {
                    // Assume that lights don't have any children that are also lights
                    turnOnZoneLights(child);
                }
            }
	    }


	    // A utility class to aid in sorting scene nodes into a distance order
	    class LightDistanceElement : IComparable<LightDistanceElement>
	    {
    	    public LightDistanceElement() {}

		    public LightDistanceElement(ILightSceneNode n, float d)
            {
			    node = n;
                distance = d;
            }

		    public ILightSceneNode node;
		    public float distance;

            #region IComparable<LightDistanceElement> メンバ

            public int CompareTo(LightDistanceElement other)
            {
                return (int)(- other.distance + distance);
            }

            #endregion
        };
    }
}
