//
// Inclusion of standard header file
//
#include <cerrno>
#include <cmath>
#include <cstdio>
#include <cstddef>
#include <ctime>
#include <string>
#include <string.h>
#include <sstream>
#include <stdexcept>

//
// Inclusion of system header file
//
#include <sys/stat.h>
#include <sys/types.h>
#include <signal.h>
#include <unistd.h>
#include <utime.h>

//
// Inclusion of library header file
//
#include <boost/filesystem/path.hpp>

//
// Inclusion of local header file
//
#include "fatal_error.hpp"
#include "islibfunc.hpp"
#include "system_error.hpp"

namespace islib
  {
    //
    // Definition of function
    //
    void
    link_
      (
        boost::filesystem::path const &source_file_path,
        boost::filesystem::path const &destination_file_path
      )
      {
        class local
          {
          public:
            local
              (
                boost::filesystem::path const &source_file_path_,
                boost::filesystem::path const &destination_file_path_
              )
              {
                throw_if ( std::invalid_argument ( "islib::link_: source_file_path_.empty ()" ), source_file_path_.empty () );
                throw_if ( std::invalid_argument ( "islib::link_: destination_file_path_.empty ()" ), destination_file_path_.empty () );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( source_file_path, destination_file_path );

        int link_return_value = link ( source_file_path.string ().c_str (), destination_file_path.string ().c_str () );

        if ( link_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::link_: link: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
    localtime_r_
      (
        std::time_t const * const time,
        struct std::tm * const time_information
      )
      {
        class local
          {
          public:
            local
              (
                std::time_t const * const time_,
                struct std::tm * const time_information_
              )
              {
                throw_if ( std::invalid_argument ( "islib::localtime_r_: !time_" ), !time_ );
                throw_if ( std::invalid_argument ( "islib::localtime_r_: !time_information_" ), !time_information_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( time, time_information );

        struct tm * const localtime_r_return_value = localtime_r ( time, time_information );

        if ( localtime_r_return_value == 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::localtime_r_: std::localtime_r: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
    mkdir_
      (
        boost::filesystem::path const &directory_path,
        mode_t mode
      )
      {
        class local
          {
          public:
            local
              (
                boost::filesystem::path const &directory_path_
              )
              {
                throw_if ( std::invalid_argument ( "islib::mkdir_: directory_path_.empty ()" ), directory_path_.empty () );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( directory_path );

        int mkdir_return_value = mkdir ( directory_path.string ().c_str (), mode );

        if ( mkdir_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::mkdir_: mkdir: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
    raise_
      (
        int const signal_number
      )
      {
        class local
          {
          public:
            local
              (
                int const signal_number_
              )
              {
                throw_if ( std::invalid_argument ( "islib::raise_: signal_number_ < 1 || _NSIG <= signal_number_" ), signal_number_ < 1 || _NSIG <= signal_number_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( signal_number );

        int const raise_return_value = raise ( signal_number );

        if ( raise_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::raise_: raise: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
    raise_fatal
      (
        int const signal_number
      )
      {
        class local
          {
          public:
            local
              (
                int const signal_number_
              )
              {
                throw_if ( fatal_error ( "islib::raise_fatal: signal_number_ < 1 || _NSIG <= signal_number_" ), signal_number_ < 1 || _NSIG <= signal_number_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( signal_number );

        int const raise_return_value = raise ( signal_number );

        if ( raise_return_value != 0 )
          throw_ ( fatal_error ( "islib::raise_fatal: raise: fatal error." ) );
      }

    //
    // Definition of function
    //
    void
    sigaction_
      (
        int const signal_number,
        struct sigaction const * const signal_action,
        struct sigaction * const old_signal_action
      )
      {
        class local
          {
          public:
            local
              (
                int const signal_number_,
                struct sigaction const * const signal_action_
              )
              {
                throw_if ( std::invalid_argument ( "islib::sigaction_: signal_number_ < 1 || _NSIG <= signal_number_" ), signal_number_ < 1 || _NSIG <= signal_number_ );
                throw_if ( std::invalid_argument ( "islib::sigaction_: !signal_action_" ), !signal_action_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( signal_number, signal_action );

        int const sigaction_return_value = sigaction ( signal_number, signal_action, old_signal_action );

        if ( sigaction_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::sigaction_: sigaction: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
    sigaction_fatal
      (
        int const signal_number,
        struct sigaction const * const signal_action,
        struct sigaction * const old_signal_action
      )
      {
        class local
          {
          public:
            local
              (
                int const signal_number_,
                struct sigaction const * const signal_action_
              )
              {
                throw_if ( fatal_error ( "islib::sigaction_fatal: signal_number_ < 1 || _NSIG <= signal_number_" ), signal_number_ < 1 || _NSIG <= signal_number_ );
                throw_if ( fatal_error ( "islib::sigaction_fatal: !signal_action_" ), !signal_action_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( signal_number, signal_action );

        int const sigaction_return_value = sigaction ( signal_number, signal_action, old_signal_action );

        if ( sigaction_return_value != 0 )
          throw_ ( fatal_error ( "islib::sigaction_fatal: sigaction: fatal error." ) );
      }

    //
    // Definition of function
    //
    void
    sigfillset_
      (
        sigset_t * const signal_set
      )
      {
        class local
          {
          public:
            local
              (
                sigset_t * const signal_set_
              )
              {
                throw_if ( std::invalid_argument ( "islib::sigfillset_: !signal_set_" ), !signal_set_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( signal_set );

        int const sigfillset_return_value = sigfillset ( signal_set );

        if ( sigfillset_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::sigfillset_: sigfillset: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
    sigfillset_fatal
      (
        sigset_t * const signal_set
      )
      {
        class local
          {
          public:
            local
              (
                sigset_t * const signal_set_
              )
              {
                throw_if ( fatal_error ( "islib::sigfillset_fatal: !signal_set_" ), !signal_set_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( signal_set );

        int const sigfillset_return_value = sigfillset ( signal_set );

        if ( sigfillset_return_value != 0 )
          throw_ ( fatal_error ( "islib::sigfillset_fatal: sigfillset: fatal error." ) );
      }

    //
    // Definition of function
    //
    void
    sigprocmask_
      (
        int const mode,
        sigset_t const * const signal_set,
        sigset_t * const old_signal_set
      )
      {
        class local
          {
          public:
            local
              (
                int const mode_,
                sigset_t const * const signal_set_
              )
              {
                throw_if ( std::invalid_argument ( "islib::sigprocmask_: mode_ != SIG_BLOCK && mode_ != SIG_UNBLOCK && mode_ != SIG_SETMASK" ), mode_ != SIG_BLOCK && mode_ != SIG_UNBLOCK && mode_ != SIG_SETMASK );
                throw_if ( std::invalid_argument ( "islib::sigprocmask_: !signal_set_" ), !signal_set_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( mode, signal_set );

        int const sigprocmask_return_value = sigprocmask ( mode, signal_set, old_signal_set );

        if ( sigprocmask_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::sigprocmask_: sigprocmask: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
    sigprocmask_fatal
      (
        int const mode,
        sigset_t const * const signal_set,
        sigset_t * const old_signal_set
      )
      {
        class local
          {
          public:
            local
              (
                int const mode_,
                sigset_t const * const signal_set_
              )
              {
                throw_if ( fatal_error ( "islib::sigprocmask_fatal: mode_ != SIG_BLOCK && mode_ != SIG_UNBLOCK && mode_ != SIG_SETMASK" ), mode_ != SIG_BLOCK && mode_ != SIG_UNBLOCK && mode_ != SIG_SETMASK );
                throw_if ( fatal_error ( "islib::sigprocmask_fatal_: !signal_set_" ), !signal_set_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( mode, signal_set );

        int const sigprocmask_return_value = sigprocmask ( mode, signal_set, old_signal_set );

        if ( sigprocmask_return_value != 0 )
          throw_ ( fatal_error ( "islib::sigprocmask_fatal: sigprocmask: fatal error." ) );
      }

    //
    // Definition of function
    //
    void
    stat_
      (
        boost::filesystem::path const &path,
        struct stat * const status_information
      )
      {
        class local
          {
          public:
            local
              (
                boost::filesystem::path const &path_,
                struct stat * const status_information_
              )
              {
                throw_if ( std::invalid_argument ( "islib::stat_: path_.empty ()" ), path_.empty () );
                throw_if ( std::invalid_argument ( "islib::stat_: !status_information_" ), !status_information_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( path, status_information );

        int const stat_return_value = stat ( path.string ().c_str (), status_information );

        if ( stat_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::stat_: stat: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    std::time_t
    time
      (
      )
      {
        std::time_t time_;

        std::time_t time_return_value = std::time ( &time_ );

        if ( time_return_value == -1 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::time: std::time: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }

        return time_;
      }

    //
    // Definition of function
    //
    void
    unlink_
      (
        boost::filesystem::path const &file_path
      )
      {
        class local
          {
          public:
            local
              (
                boost::filesystem::path const &file_path_
              )
              {
                throw_if ( std::invalid_argument ( "islib::unlink_: file_path_.empty ()" ), file_path_.empty () );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( file_path );

        int unlink_return_value = unlink ( file_path.string ().c_str () );

        if ( unlink_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::unlink_: unlink: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
    utime_
      (
        boost::filesystem::path const &path,
        struct utimbuf const * const utime_information
      )
      {
        class local
          {
          public:
            local
              (
                boost::filesystem::path const &path_,
                struct utimbuf const * const utime_information_
              )
              {
                throw_if ( std::invalid_argument ( "islib::utime_: path_.empty ()" ), path_.empty () );
                throw_if ( std::invalid_argument ( "islib::utime_: !utime_information_" ), !utime_information_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( path, utime_information );

        int utime_return_value = utime ( path.string ().c_str (), utime_information );

        if ( utime_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::utime_: utime: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
    write_
      (
        int const file_descriptor,
        void const *buffer,
        std::size_t const number_of_bytes
      )
      {
        class local
          {
          public:
            local
              (
                int const file_descriptor_,
                void const *buffer_
              )
              {
                throw_if ( std::invalid_argument ( "islib::write_: file_descriptor_ < 0" ), file_descriptor_ < 0 );
                throw_if ( std::invalid_argument ( "islib::write_: !buffer_" ), !buffer_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( file_descriptor, buffer );

        ssize_t const write_return_value = write ( file_descriptor, buffer, number_of_bytes );

        if ( write_return_value == -1 )
          {
            int const saved_errno = errno;
            char buffer_ [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::write_: write: " << strerror_r ( saved_errno, buffer_, BUFSIZ );

            throw_ ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
    write_assert
      (
        int const file_descriptor,
        void const *buffer,
        std::size_t const number_of_bytes
      )
      {
        class local
          {
          public:
            local
              (
                int const file_descriptor_,
                void const *buffer_
              )
              {
                throw_if ( fatal_error ( "islib::write_: file_descriptor_ < 0" ), file_descriptor_ < 0 );
                throw_if ( fatal_error ( "islib::write_: !buffer_" ), !buffer_ );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( file_descriptor, buffer );

        ssize_t const write_return_value = write ( file_descriptor, buffer, number_of_bytes );

        assert_ ( write_return_value != -1, "islib::write_: `write_return_value != -1' has failed." );
      }

    //
    // Definition of specialized template function
    //
    template
      <
      >
    bool
    equals_to
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return std::fabs ( x - y ) <= permissible_error;
      }

    //
    // Definition of specialized template function
    //
    template
      <
      >
    bool
    does_not_equal_to
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return std::fabs ( x - y ) > permissible_error;
      }

    //
    // Definition of specialized template function
    //
    template
      <
      >
    bool
    is_less_than
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return y - x > permissible_error;
      }

    //
    // Definition of specialized template function
    //
    template
      <
      >
    bool
    is_greater_than
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return x - y > permissible_error;
      }

    //
    // Definition of specialized template function
    //
    template
      <
      >
    bool
    is_less_than_or_equals_to
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return x - y <= permissible_error;
      }

    //
    // Definition of specialized template function
    //
    template <>
    bool
    is_greater_than_or_equals_to
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return y - x <= permissible_error;
      }
  }

//
// End of file
//
