/*======================================================================
----------------------------------------------------------------------*/
/**
 * @file		iutest_case_c.h
 * @brief		iris unit test case t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2012 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see LICENSE
*/
/*----------------------------------------------------------------------
======================================================================*/
#ifndef INCG_IRIS_iutest_case_c_H_1EFD40F6_D189_467b_93B0_887A64E2B830_
#define INCG_IRIS_iutest_case_c_H_1EFD40F6_D189_467b_93B0_887A64E2B830_

/* include ===========================================================*/
#include "iutest_info_c.h"

/* typedef ===========================================================*/
/**
 * @brief	eXgP[X SetUp ֐|C^^
*/
typedef void (*iuTestSetUpTestCase)(void);

/**
 * @brief	eXgP[X TearDown ֐|C^^
*/
typedef void (*iuTestTearDownTestCase)(void);

/* enum ==============================================================*/
typedef enum IUTESTCASE_FLAG
{
	IUTESTCASE_DISABLED			= 0x00000002,	/*!< ǂ */
} IUTESTCASE_FLAG;

/* struct ============================================================*/
/**
 * @brief	TestCase \
*/
typedef struct iuTestCase_t
{
	const char* name;		/*!< eXgP[X */
	iuTestInfo*	list;		/*!< TestInfo Xg */
	iuTestSetUpTestCase		setup;		/*!< SetUp */
	iuTestTearDownTestCase	teardown;	/*!< TearDown */
	iuTimeInMillisec		elapsedmsec;/*!< s */
	iuUInt32	flag;		/*!< bit tO */
	struct iuTestCase_t	*next;		/*!<  TestCase ̃m[h */
} iuTestCase;

/* declare ===========================================================*/
static int iuTestCase_GetTotalTestCount(iuTestCase* test_case);
static int iuTestCase_GetTestToRunCount(iuTestCase* test_case);

/* function ==========================================================*/
/**
 * @brief	eXg̑擾
*/
static int iuTestCase_GetTotalTestCount(iuTestCase* test_case)
{
	int cnt = 0;
	iuTestHelper_CountList(cnt, iuTestInfo, test_case->list);
	return cnt;
}

/**
 * @brief	sΏۂ̃eXg̑擾
*/
static int iuTestCase_GetTestToRunCount(iuTestCase* test_case)
{
	int cnt = 0;
	iuTestInfo* curr = test_case->list;
	while( curr != NULL )
	{
		if( iuTestInfo_IsShouldRunTest(curr) )
		{
			++cnt;
		}
		curr = curr->next;
	}
	return cnt;
}

/**
 * @brief	eXg̑擾
*/
static int iuTestCase_GetSuccessfulTestCount(iuTestCase* test_case)
{
	int cnt = 0;
	iuTestInfo* curr = test_case->list;
	while( curr != NULL )
	{
		if( iuTestInfo_IsShouldRunTest(curr) && !iuTestInfo_HasFailure(curr) )
		{
			++cnt;
		}
		curr = curr->next;
	}
	return cnt;
}

/**
 * @brief	seXg̑擾
*/
static int iuTestCase_GetFailureTestCount(iuTestCase* test_case)
{
	int cnt = 0;
	iuTestInfo* curr = test_case->list;
	while( curr != NULL )
	{
		if( iuTestInfo_IsShouldRunTest(curr) && iuTestInfo_HasFailure(curr) )
		{
			++cnt;
		}
		curr = curr->next;
	}
	return cnt;
}

/**
 * @brief	eXgʂ̎擾
*/
static iuBOOL iuTestCase_IsFaild(iuTestCase* test_case)
{
	return iuTestCase_GetFailureTestCount(test_case) != 0 ? TRUE : FALSE;
}

/**
 * @brief	ǂ
*/
static iuBOOL iuTestCase_IsDisabledTest(iuTestCase* test_case)
{
	return test_case->flag & IUTESTINFO_DISABLED ? TRUE : FALSE;
}

/**
 * @brief	eXg̑擾
*/
static int iuTestCase_GetDisableTestCount(iuTestCase* test_case)
{
	if( iuTestCase_IsDisabledTest(test_case) )
	{
		return iuTestCase_GetTotalTestCount(test_case);
	}
	else
	{
		int cnt = 0;
		iuTestInfo* curr = test_case->list;
		while( curr != NULL )
		{
			if( iuTestInfo_IsDisabledTest(curr) )
			{
				++cnt;
			}
			curr = curr->next;
		}
		return cnt;
	}
}

/**
 * @brief	XLbveXg̑擾
*/
static int iuTestCase_GetSkippedTestCount(iuTestCase* test_case)
{
	int cnt = 0;
	iuTestInfo* curr = test_case->list;
	while( curr != NULL )
	{
		if( !iuTestInfo_IsRanTest(curr) )
		{
			++cnt;
		}
		curr = curr->next;
	}
	return cnt;
}

/**
 * @brief	seXg邩ǂ
*/
static iuBOOL iuTestCase_HasShouldRunTest(iuTestCase* test_case)
{
	iuTestInfo* curr = test_case->list;
	while( curr != NULL )
	{
		if( iuTestInfo_IsShouldRunTest(curr) )
		{
			return TRUE;
		}
		curr = curr->next;
	}
	return FALSE;
}

/**
 * @brief	Ԃ̃NA
*/
static void iuTestCase_Clear(iuTestCase* test_case)
{
	iuTestInfo* curr = test_case->list;
	while( curr != NULL )
	{
		iuTestInfo_Clear(curr);
		curr = curr->next;
	}
}

/**
 * @brief	eXg̃tB^O
*/
static void iuTestCase_Filter(iuTestCase* test_case)
{
	iuTestInfo* curr = test_case->list;
	while( curr != NULL )
	{
		if( iuTestCase_IsDisabledTest(test_case) )
		{
			/* DISABLE ̓` */
			curr->flag |= IUTESTINFO_DISABLED;
		}
		iuTestInfo_Filter(curr);
		curr = curr->next;
	}
}

/**
 * @brief	TestInfo ̒ǉ
*/
static IUTEST_ATTRIBUTE_UNUSED_ int	iuTestCase_AddTestInfo(iuTestCase* test_case, iuTestInfo* test_info)
{
	if( test_case == NULL ) return FALSE;
	if( test_info == NULL ) return FALSE;

	{
		iuBOOL find = FALSE;
		iuTestHelper_FindList(iuTestInfo, test_case->list, test_info, find);
		/* łɒǉς */
		if( find ) return TRUE;
	}
	iuTestHelper_AddList(iuTestInfo, test_case->list, test_info);

	if( iuString_IsDisabledTestName(test_info->name) )
	{
		test_info->flag |= IUTESTINFO_DISABLED;
	}
	test_info->testcase_name = test_case->name;
	return TRUE;
}

/**
 * @brief	eXg̎s
*/
static iuBOOL	iuTestCase_Run(iuTestCase* test_case)
{
	iuBOOL ret = TRUE;
	if( test_case == NULL ) return FALSE;
	if( iuTestCase_GetTestToRunCount(test_case) == 0 ) return TRUE;

	iuTestEnv_ListenerEvent_OnTestCaseStart(test_case);

	if( iuTestEnv_IsEnableShuffleTests() )
	{
		iuTestHelper_ShuffleList(iuTestInfo, iuTestRandom_Gen(&IIUT_C_TESTENV().random_context)
			, (unsigned int)iuTestCase_GetTotalTestCount(test_case), test_case->list);
	}

	{
		iuTestStopWatch watch;
		iuTest_StopWatchStart(&watch);

		if( test_case->setup != NULL )
		{
			(*test_case->setup)();
		}
		{
			iuTestInfo* curr = test_case->list;
			while( curr != NULL )
			{
				if( !iuTestInfo_Run(test_case, curr) ) ret = FALSE;
				curr = curr->next;
			}
		}
		if( test_case->teardown != NULL )
		{
			(*test_case->teardown)();
		}

		test_case->elapsedmsec = iuTest_StopWatchStop(&watch);
	}

	iuTestEnv_ListenerEvent_OnTestCaseEnd(test_case);
	return ret;
}

#endif
