package jp.sourceforge.functional;

import jp.sourceforge.functional.converter.UnchangeConverter;
import jp.sourceforge.functional.pair.Pair;
import jp.sourceforge.functional.util.ConvertFunctors;

/**
 * <p>２つの引数を取るコンバーター。
 * </p>
 * <p>
 * <code>UnaryConverter</code><code>Pair</code>によって拡張（{@code extends
 * UnaryConverter<Pair<T1, T2>>}）しているため、 通常の（引数がひとつ以下の）ファンクタと同様に扱うことが出来ます。
 * </p>
 * <p>
 * このクラスのオブジェクトを引数として取りたい場合は、パラメータの型を
 * 
 * <pre>
 * Converter&lt;? super Pair&lt;? extends T1, ? extends T2&gt;&gt;
 * </pre>
 * 
 * と宣言することを推奨します。<BR>
 * それによって、メソッドは<code>BinaryConverter</code>だけではなく、<code>Object</code>を引数に取る
 * <code>UnaryConverter</code>や、<code>NullaryConverter</code>
 * なども受け取れるようになり、汎用性を大幅に高めることが出来ます。
 * </p>
 * 
 * @author Fujii Kenichi
 * 
 * @param <T1>
 *            <code>Pair.first</code>の型
 * @param <T2>
 *            <code>Pair.second</code>の型
 * @param <R>
 *            コンバーターの戻り値
 */
public abstract class BinaryConverter<T1, T2, R> extends
		UnaryConverter<Pair<? extends T1, ? extends T2>, R> {

	public abstract R convert(T1 first, T2 second);

	@Override
	public R convert(Pair<? extends T1, ? extends T2> pair) {
		return convert(pair.first, pair.second);
	}

	public UnaryConverter<T2, R> bindFirst(final T1 first) {
		return new UnaryConverter<T2, R>() {
			@Override
			public R convert(T2 second) {
				return BinaryConverter.this.convert(first, second);
			}
		};
	}

	public UnaryConverter<T1, R> bindSecond(final T2 second) {
		return new UnaryConverter<T1, R>() {
			@Override
			public R convert(T1 first) {
				return BinaryConverter.this.convert(first, second);
			}
		};
	}

	public UnaryConverter<T1, ? extends UnaryConverter<T2, R>> firstBinder() {
		return ConvertFunctors.<T1, T2, R> bindFirst().bindFirst(this);
	}

	public UnaryConverter<T2, ? extends UnaryConverter<T1, R>> secondBinder() {
		return ConvertFunctors.<T1, T2, R> bindSecond().bindFirst(this);
	}

	public <S1, S2> BinaryConverter<S1, S2, R> arrangeBoth(
			final Converter<? super S1, ? extends T1> first_converter,
			final Converter<? super S2, ? extends T2> second_converter) {
		return new BinaryConverter<S1, S2, R>() {
			@Override
			public R convert(S1 first, S2 second) {
				return BinaryConverter.this.convert(first_converter
						.convert(first), second_converter.convert(second));
			}
		};
	}

	public <S> BinaryConverter<S, T2, R> arrangeFirst(
			final Converter<? super S, ? extends T1> converter) {
		return arrangeBoth(converter, UnchangeConverter.<T2> getInstance());
	}

	public <S> BinaryConverter<T1, S, R> arrangeSecond(
			final Converter<? super S, ? extends T2> converter) {
		return arrangeBoth(UnchangeConverter.<T1> getInstance(), converter);
	}

	@Override
	public <RESULT> BinaryConverter<T1, T2, RESULT> arrangeResult(
			final Converter<? super R, ? extends RESULT> result_converter) {
		return new BinaryConverter<T1, T2, RESULT>() {
			@Override
			public RESULT convert(T1 first, T2 second) {
				return result_converter.convert(BinaryConverter.this.convert(
						first, second));
			}
		};
	}
}
