package jp.sourceforge.functional;

import jp.sourceforge.functional.pair.Pair;
import jp.sourceforge.functional.util.ConvertFunctors;

/**
 * <p>
 * <code>Converter</code>インターフェースの抽象実装クラス。<BR>
 * <code>bind()</code>、<code>arrange()</code> など、汎用性の高いメソッドを提供します。<BR>
 * </p>
 * このパッケージで定義されたファンクタクラスのほとんど（<code>NullaryConverter</code>及び派生クラスを除く）は、 この
 * <code>UnaryConverter</code>
 * を拡張しており、ファンクタクラスを設計する際には同じようにこのクラス（またはその派生クラス）から拡張することを推奨します。</p>
 * 
 * @author Fujii Kenichi
 * 
 * @param <T>
 *            引数の型
 * @param <R>
 *            戻り値の型
 */
public abstract class UnaryConverter<T, R> implements Converter<T, R> {

	/**
	 * <p>
	 * 引数をバインド（カリー化）し、引数を持たないファンクタ（<code>NullaryConverter</code>）を生成します。<BR>
	 * （”カリー化”が解らない人は<em>”関数型言語”</em>で調べてみると良いです。）
	 * </p>
	 * <p>
	 * これは<em>「引数の値を遅延評価する」</em>と言うことなので、可変オブジェクトを渡す際には注意が必要です。
	 * </p>
	 * 
	 * @param bind
	 *            バインドするオブジェクト
	 * @return カリー化された<code>NullaryConverter</code>
	 */
	public NullaryConverter<R> bind(final T bind) {
		return new NullaryConverter<R>() {
			@Override
			public R get() {
				return UnaryConverter.this.convert(bind);
			}
		};
	}

	/**
	 * <p>
	 * <code>bind()</code>メソッドを表すファンクタを返す。
	 * </p>
	 * <p>
	 * これは関数型言語における <em>『N個の引数をとる関数』は、『1個の引数を取り【N-1個の引数を取る関数】を返す関数』と等価である</em>
	 * と言う （ややこしい）概念を再現するためのものです。
	 * </p>
	 * 
	 * @return <code>T</code>を受け取り、{@code NullaryConverter<R>}を返すファンクタ。
	 */
	public UnaryConverter<T, ? extends NullaryConverter<R>> binder() {
		return ConvertFunctors.<T, R> bind().bindFirst(this);
	}

	/**
	 * <p>
	 * 引数を変更したファンクタを返します。<BR>
	 * より正確には、 <em>変更後の引数から現在（変更前）の引数に変換するコンバーター </em> によって、ファンクタをラッピングします。
	 * </p>
	 * <p>
	 * ファンクタ同士のバケツリレーを想像してもらうと、解りやすいかも知れません。
	 * </p>
	 * 
	 * @param <S>
	 *            新しい（変換前の）引数
	 * @param converter
	 *            引数を変換するためのコンバーター
	 * @return 変更後のファンクタ
	 */
	public <S> UnaryConverter<S, R> arrange(
			final Converter<? super S, ? extends T> converter) {
		return new UnaryConverter<S, R>() {
			@Override
			public R convert(S element) {
				return UnaryConverter.this.convert(converter.convert(element));
			}
		};
	}

	/**
	 * <p>
	 * <code>Pair</code>を引数とするコンバーターでラッピングすることによって、<code>BinaryConverter</code>
	 * を生成します。
	 * </p>
	 * 
	 * @param <S1>
	 *            <code>Pair.first</code>の型
	 * @param <S2>
	 *            <code>Pair.second</code>の型
	 * @param converter
	 *            {@code Pair<S1, S2>}を引数に取る（取ることが出来る）コンバーター
	 * @return 変更後の<code>BinaryConverter</code>
	 */
	public <S1, S2> BinaryConverter<S1, S2, R> arrangeToBinary(
			final Converter<? super Pair<? extends S1, ? extends S2>, ? extends T> converter) {
		return new BinaryConverter<S1, S2, R>() {
			@Override
			public R convert(S1 first, S2 second) {
				return UnaryConverter.this.convert(converter.convert(Pair.makePair(
						first, second)));
			}
		};
	}

	/**
	 * <p>
	 * 戻り値を変換したコンバーターを返します。
	 * </p>
	 * <p>
	 * <code>arrange()</code>メソッドは、値を”渡して”くれるコンバーターによって変換を行う処理でしたが、
	 * こちらは値を”受け取って”くれるコンバーターによって変換を行います。
	 * </p>
	 * 
	 * @param <RESULT>
	 *            新しい（変換後の）戻り値
	 * @param result_converter
	 *            戻り値を変換するコンバーター
	 * @return 変換後のコンバーター
	 */
	public <RESULT> UnaryConverter<T, RESULT> arrangeResult(
			final Converter<? super R, ? extends RESULT> result_converter) {
		return new UnaryConverter<T, RESULT>() {
			@Override
			public RESULT convert(T element) {
				return result_converter.convert(UnaryConverter.this
						.convert(element));
			}
		};
	}
}
