package jp.sourceforge.functional.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import jp.sourceforge.functional.Classifier;
import jp.sourceforge.functional.Converter;
import jp.sourceforge.functional.Visitor;
import jp.sourceforge.functional.iterable.ConvertIterable;
import jp.sourceforge.functional.iterable.ExtractIterable;
import jp.sourceforge.functional.pair.SamePair;

/**
 * ファンクタによる巡回をサポートするメソッドを集めたユーティリティークラス。
 * 
 * @author Fujii Kenichi
 * 
 */
public class Around {

	public static <T> void visit(Iterable<? extends T> targets,
			Visitor<? super T> visitor) {
		for (T target : targets) {
			visitor.visit(target);
		}
	}

	public static <SOURCE, T> Iterable<T> convertIterable(
			Iterable<? extends SOURCE> targets,
			Converter<? super SOURCE, ? extends T> converter) {
		return new ConvertIterable<SOURCE, T>(targets, converter);
	}

	public static <SOURCE, T, C extends Collection<? super T>> C convertCollection(
			C initial, Iterable<? extends SOURCE> targets,
			Converter<? super SOURCE, ? extends T> converter) {
		for (SOURCE source : targets) {
			initial.add(converter.convert(source));
		}
		return initial;
	}

	public static <SOURCE, T> Set<T> convertSet(
			Iterable<? extends SOURCE> targets,
			Converter<? super SOURCE, ? extends T> converter) {
		return convertCollection(new HashSet<T>(), targets, converter);
	}

	public static <SOURCE, T> List<T> convertList(
			Iterable<? extends SOURCE> targets,
			Converter<? super SOURCE, ? extends T> converter) {
		return convertCollection(new ArrayList<T>(), targets, converter);
	}

	public static <T> Iterable<T> extractIterable(
			Iterable<? extends T> targets, Classifier<? super T> classifier) {
		return new ExtractIterable<T>(targets, classifier);
	}

	public static <T, C extends Collection<? super T>> C extractCollection(
			C initial, Iterable<? extends T> targets,
			Classifier<? super T> classifier) {
		for (T e : targets) {
			if (classifier.classify(e)) initial.add(e);
		}
		return initial;
	}

	public static <T> Set<T> extractSet(Iterable<? extends T> targets,
			Classifier<? super T> classifier) {
		return extractCollection(new HashSet<T>(), targets, classifier);
	}

	public static <T> List<T> extractList(Iterable<? extends T> targets,
			Classifier<? super T> classifier) {
		return extractCollection(new ArrayList<T>(), targets, classifier);
	}

	// true elements => first. false elements => second
	public static <E> SamePair<List<E>> divideList(Iterable<E> target,
			Classifier<? super E> classifier) {
		List<E> true_elements = new ArrayList<E>();
		List<E> false_elements = new ArrayList<E>();
		for (E e : target) {
			if (classifier.classify(e)) true_elements.add(e);
			else false_elements.add(e);
		}
		return new SamePair<List<E>>(true_elements, false_elements);
	}

	// true elements => first. false elements => second
	public static <E> SamePair<Set<E>> divideSet(Iterable<E> target,
			Classifier<? super E> classifier) {
		Set<E> true_elements = new HashSet<E>();
		Set<E> false_elements = new HashSet<E>();
		for (E e : target) {
			if (classifier.classify(e)) {
				true_elements.add(e);
			}
			else false_elements.add(e);
		}
		return new SamePair<Set<E>>(true_elements, false_elements);
	}
}
