package jp.sourceforge.functional.util;

import java.util.Collection;
import java.util.Iterator;

import jp.sourceforge.functional.BinaryClassifier;
import jp.sourceforge.functional.Classifier;
import jp.sourceforge.functional.Converter;
import jp.sourceforge.functional.NullaryClassifier;
import jp.sourceforge.functional.UnaryClassifier;
import jp.sourceforge.functional.pair.BoundPairMaker;
import jp.sourceforge.functional.pair.Pair;

public class ClassifyUtil {

	public static <T> boolean necessaryClassify(
			Collection<Classifier<? super T>> classifiers, T object) {
		for (Classifier<? super T> classifier : classifiers) {
			if (!classifier.classify(object)) return false;
		}
		return true;
	}

	public static <T> boolean sufficientClassify(
			Collection<Classifier<? super T>> classifiers, T object) {
		for (Classifier<? super T> type : classifiers) {
			if (type.classify(object)) return true;
		}
		return false;
	}

	public static <T> T findFirst(Iterable<? extends T> target,
			Classifier<? super T> classifier) {
		for (T element : target) {
			if (classifier.classify(element)) {
				return element;
			}
		}
		return null;
	}

	public static <T> boolean isWhichOf(Iterable<T> target,
			Classifier<? super T> classifier) {
		for (T o : target) {
			if (classifier.classify(o)) return true;
		}
		return false;
	}

	public static <T> boolean isAllOf(Iterable<? extends T> target,
			Classifier<? super T> classifier) {
		for (T o : target) {
			if (!classifier.classify(o)) return false;
		}
		return true;
	}

	public static <T> int countTrue(Iterable<? extends T> target,
			final Classifier<? super T> classifier) {
		int result = 0;
		for (T object : target) {
			if (classifier.classify(object)) result++;
		}
		return result;
	}

	public static <T> void removeIf(Iterable<? extends T> iterable,
			Classifier<? super T> classifier) {
		Iterator<? extends T> iterator = iterable.iterator();
		while (iterator.hasNext()) {
			if (classifier.classify(iterator.next())) {
				iterator.remove();
			}
		}
	}

	public static <T> NullaryClassifier bind(
			final Classifier<? super T> classifier, final T bind) {
		return new NullaryClassifier() {
			@Override
			public boolean conclude() {
				return classifier.classify(bind);
			}
		};
	}

	public static <T1, T2> UnaryClassifier<T2> bindFirst(
			Classifier<? super Pair<? extends T1, ? extends T2>> classifier,
			T1 first) {
		return arrange(classifier, PairFunctors.<T1, T2> make()
				.bindFirst(first));
	}

	public static <T1, T2> UnaryClassifier<T1> bindSecond(
			Classifier<? super Pair<? extends T1, ? extends T2>> classifier,
			T2 second) {
		return arrange(classifier, PairFunctors.<T1, T2> make().bindSecond(
				second));
	}

	public static <T1, T2> NullaryClassifier bindBoth(
			Classifier<? super Pair<? extends T1, ? extends T2>> classifier,
			T1 first, T2 second) {
		return bind(classifier, Pair.makePair(first, second));
	}

	public static <E1, E2, BIND, REMAIN> UnaryClassifier<REMAIN> bindSide(
			Classifier<? super Pair<? extends E1, ? extends E2>> classifier,
			Pair.Side<E1, E2, BIND, REMAIN> side, BIND bind) {
		return arrange(classifier, new BoundPairMaker<E1, E2, BIND, REMAIN>(
				side, bind));
	}

	public static <S, T> UnaryClassifier<S> arrange(
			final Classifier<? super T> classifier,
			final Converter<? super S, ? extends T> converter) {
		return new UnaryClassifier<S>() {
			@Override
			public boolean classify(S source) {
				return classifier.classify(converter.convert(source));
			}
		};
	}

	public static <S1, S2, T> BinaryClassifier<S1, S2> arrangeToBinary(
			final Classifier<? super T> classifier,
			final Converter<? super Pair<? extends S1, ? extends S2>, ? extends T> pair_converter) {
		return new BinaryClassifier<S1, S2>() {
			@Override
			public boolean classify(S1 first, S2 second) {
				return classifier.classify(pair_converter.convert(Pair.makePair(
						first, second)));
			}
		};
	}

	public static <S1, S2, T1, T2> BinaryClassifier<S1, S2> arrangeBoth(
			Classifier<? super Pair<? extends T1, ? extends T2>> classifier,
			Converter<? super S1, ? extends T1> first_converter,
			Converter<? super S2, ? extends T2> second_converter) {
		return arrangeToBinary(classifier, PairFunctors.<T1, T2> make()
				.arrangeBoth(first_converter, second_converter));
	}

	public static <S1, T1, T2> BinaryClassifier<S1, T2> arrangeFirst(
			Classifier<? super Pair<? extends T1, ? extends T2>> classifier,
			Converter<? super S1, ? extends T1> first_converter) {
		return arrangeToBinary(classifier, PairFunctors.<T1, T2> make()
				.arrangeFirst(first_converter));
	}

	public static <S2, T1, T2> BinaryClassifier<T1, S2> arrangeSecond(
			Classifier<? super Pair<? extends T1, ? extends T2>> classifier,
			Converter<? super S2, ? extends T2> second_converter) {
		return arrangeToBinary(classifier, PairFunctors.<T1, T2> make()
				.arrangeSecond(second_converter));
	}

	public static <T> UnaryClassifier<T> makeClassifier(
			final Classifier<? super T> classifier) {
		return new UnaryClassifier<T>() {
			@Override
			public boolean classify(T element) {
				return classifier.classify(element);
			}
		};
	}

	public static <T> UnaryClassifier<T> makeClassifier(
			final Converter<? super T, ? extends Boolean> converter) {
		return new UnaryClassifier<T>() {
			@Override
			public boolean classify(T element) {
				return converter.convert(element);
			}
		};
	}

	public static <T1, T2> BinaryClassifier<T1, T2> makeBinaryClassifier(
			final Classifier<? super Pair<? extends T1, ? extends T2>> classifier) {
		return new BinaryClassifier<T1, T2>() {
			@Override
			public boolean classify(Pair<? extends T1, ? extends T2> pair) {
				return classifier.classify(pair);
			}

			@Override
			public boolean classify(T1 first, T2 second) {
				return classify(Pair.makePair(first, second));
			}

		};
	}

	public static <T1, T2> BinaryClassifier<T1, T2> makeBinaryClassifier(
			final Converter<? super Pair<? extends T1, ? extends T2>, ? extends Boolean> converter) {
		return new BinaryClassifier<T1, T2>() {
			@Override
			public boolean classify(T1 first, T2 second) {
				return converter.convert(Pair.makePair(first, second));
			}
		};
	}

	public static <T> UnaryClassifier<T> negative(
			final Classifier<? super T> classifier) {
		return new UnaryClassifier<T>() {
			@Override
			public boolean classify(T object) {
				return !classifier.classify(object);
			}
		};
	}
}
