package jp.sourceforge.functional.util;

import java.util.HashSet;
import java.util.Set;

import jp.sourceforge.functional.BinaryConverter;
import jp.sourceforge.functional.Classifier;
import jp.sourceforge.functional.Converter;
import jp.sourceforge.functional.NullaryConverter;
import jp.sourceforge.functional.UnaryConverter;
import jp.sourceforge.functional.Visitor;
import jp.sourceforge.functional.pair.BoundPairMaker;
import jp.sourceforge.functional.pair.Pair;

public class ConvertUtil {

	public static <T, R> NullaryConverter<R> bind(
			final Converter<? super T, ? extends R> converter, final T bind) {
		return new NullaryConverter<R>() {
			@Override
			public R get() {
				return converter.convert(bind);
			}
		};
	}

	public static <T1, T2, R> UnaryConverter<T2, R> bindFirst(
			Converter<? super Pair<? extends T1, ? extends T2>, ? extends R> converter,
			T1 first) {
		return arrange(converter, PairFunctors.<T1, T2> make().bindFirst(first));
	}

	public static <T1, T2, R> UnaryConverter<T1, R> bindSecond(
			Converter<? super Pair<? extends T1, ? extends T2>, ? extends R> converter,
			T2 second) {
		return arrange(converter, PairFunctors.<T1, T2> make().bindSecond(
				second));
	}

	public static <T1, T2, R> NullaryConverter<R> bindBoth(
			Converter<? super Pair<? extends T1, ? extends T2>, ? extends R> converter,
			T1 first, T2 second) {
		return bind(converter, Pair.makePair(first, second));
	}

	public static <E1, E2, BIND, REMAIN, R> UnaryConverter<REMAIN, R> bindSide(
			Converter<? super Pair<? extends E1, ? extends E2>, R> converter,
			Pair.Side<E1, E2, BIND, REMAIN> side, BIND bind) {
		return arrange(converter, new BoundPairMaker<E1, E2, BIND, REMAIN>(
				side, bind));
	}

	public static <T1, T2, R> Set<UnaryConverter<T2, R>> bindFirsts(
			Converter<? super Pair<? extends T1, ? extends T2>, ? extends R> converter,
			Iterable<? extends T1> firsts) {
		Set<UnaryConverter<T2, R>> result = new HashSet<UnaryConverter<T2, R>>();
		for (T1 first : firsts) {
			result.add(bindFirst(converter, first));
		}
		return result;
	}

	public static <T1, T2, R> Set<UnaryConverter<T1, R>> bindSeconds(
			Converter<? super Pair<? extends T1, ? extends T2>, ? extends R> converter,
			Iterable<? extends T2> seconds) {
		Set<UnaryConverter<T1, R>> result = new HashSet<UnaryConverter<T1, R>>();
		for (T2 second : seconds) {
			result.add(bindSecond(converter, second));
		}
		return result;
	}

	public static <S, T, R> UnaryConverter<S, R> arrange(
			final Converter<? super T, ? extends R> converter,
			final Converter<? super S, ? extends T> source_converter) {
		return new UnaryConverter<S, R>() {
			@Override
			public R convert(S source) {
				return converter.convert(source_converter.convert(source));
			}
		};
	}

	public static <S1, S2, T, R> BinaryConverter<S1, S2, R> arrangeToBinary(
			final Converter<? super T, ? extends R> converter,
			final Converter<? super Pair<? extends S1, ? extends S2>, ? extends T> pair_converter) {
		return new BinaryConverter<S1, S2, R>() {
			@Override
			public R convert(S1 first, S2 second) {
				return converter.convert(pair_converter.convert(Pair.makePair(
						first, second)));
			}
		};
	}

	public static <S1, S2, T1, T2, R> BinaryConverter<S1, S2, R> arrangeBoth(
			Converter<? super Pair<? extends T1, ? extends T2>, ? extends R> converter,
			Converter<? super S1, ? extends T1> first_converter,
			Converter<? super S2, ? extends T2> second_converter) {
		return arrangeToBinary(converter, PairFunctors.<T1, T2> make()
				.arrangeBoth(first_converter, second_converter));
	}

	public static <S1, T1, T2, R> BinaryConverter<S1, T2, R> arrangeFirst(
			Converter<? super Pair<? extends T1, ? extends T2>, ? extends R> converter,
			Converter<? super S1, ? extends T1> first_converter) {
		return arrangeToBinary(converter, PairFunctors.<T1, T2> make()
				.arrangeFirst(first_converter));
	}

	public static <S2, T1, T2, R> BinaryConverter<T1, S2, R> arrangeSecond(
			Converter<? super Pair<? extends T1, ? extends T2>, ? extends R> converter,
			Converter<? super S2, ? extends T2> second_converter) {
		return arrangeToBinary(converter, PairFunctors.<T1, T2> make()
				.arrangeSecond(second_converter));
	}

	public static <T, R> UnaryConverter<T, R> makeConverter(
			final Converter<? super T, ? extends R> converter) {
		return new UnaryConverter<T, R>() {
			@Override
			public R convert(T element) {
				return converter.convert(element);
			}
		};
	}

	public static <T> UnaryConverter<T, Boolean> makeConverter(
			final Classifier<? super T> classifier) {
		return new UnaryConverter<T, Boolean>() {
			@Override
			public Boolean convert(T element) {
				return classifier.classify(element);
			}
		};
	}

	public static <T> UnaryConverter<T, Void> makeConverter(
			final Visitor<? super T> visitor) {
		return new UnaryConverter<T, Void>() {
			@Override
			public Void convert(T element) {
				visitor.visit(element);
				return null;
			}
		};
	}

	public static <T1, T2, R> BinaryConverter<T1, T2, R> makeBinaryConverter(
			final Converter<Pair<? extends T1, ? extends T2>, R> converter) {
		return new BinaryConverter<T1, T2, R>() {
			@Override
			public R convert(Pair<? extends T1, ? extends T2> pair) {
				return converter.convert(pair);
			}

			@Override
			public R convert(T1 first, T2 second) {
				return convert(Pair.makePair(first, second));
			}
		};
	}

	public static <T1, T2> BinaryConverter<T1, T2, Boolean> makeBinaryConverter(
			final Classifier<Pair<? extends T1, ? extends T2>> classifier) {
		return new BinaryConverter<T1, T2, Boolean>() {
			@Override
			public Boolean convert(Pair<? extends T1, ? extends T2> pair) {
				return classifier.classify(pair);
			}

			@Override
			public Boolean convert(T1 first, T2 second) {
				return convert(Pair.makePair(first, second));
			}
		};
	}

	public static <T1, T2> BinaryConverter<T1, T2, Void> makeBinaryConverter(
			final Visitor<Pair<? extends T1, ? extends T2>> visitor) {
		return new BinaryConverter<T1, T2, Void>() {
			@Override
			public Void convert(Pair<? extends T1, ? extends T2> pair) {
				visitor.visit(pair);
				return null;
			}

			@Override
			public Void convert(T1 first, T2 second) {
				return convert(Pair.makePair(first, second));
			}
		};
	}
}
