package org.dyndns.nuda.tools.regex;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.dyndns.nuda.management.RUN_STATE;
import org.dyndns.nuda.plugin.PluginLoader;
import org.dyndns.nuda.tools.regex.annotation.Regex;
import org.dyndns.nuda.tools.regex.plugins.CachePlugin;
import org.dyndns.nuda.tools.regex.plugins.ParserPlugin;
import org.dyndns.nuda.tools.regex.plugins.ProcessorPlugin;
import org.dyndns.nuda.tools.regex.processors.AutoInputSequenceProcessor;
import org.dyndns.nuda.tools.regex.processors.InputSequenceProcessor;
import org.dyndns.nuda.tools.regex.reflection.cache.CacheManager;
import org.dyndns.nuda.tools.regex.reflection.cache.FieldCache;
import org.dyndns.nuda.tools.regex.reflection.cache.MethodCache;
import org.dyndns.nuda.tools.regex.settings.JAutoRegexerSettings;
import org.dyndns.nuda.tools.regex.util.CommonToolsVersionCheck;
import org.dyndns.nuda.tools.regex.util.RegexPatternFactory;
import org.dyndns.nuda.tools.regex.util.exception.RegexPatternException;
import org.dyndns.nuda.tools.util.StringUtil;

/**
 * 正規表現解析Bean(Regexアノテーションが付加されたBean)に対して 解析結果を提供するための正規表現プロセッサです
 * 
 * @author nkoseki
 * 
 */
public class RegexProcessor {

	/**
	 * シーケンスプロセッサ
	 */
	private static InputSequenceProcessor processor;

	private static MethodCache filterCache;

	private static FieldCache fieldCache;

	/**
	 * CommonToolsの要求バージョン
	 */
	private static final int requireCommonVersion = 1301;

	private static RUN_STATE STATE = RUN_STATE.SLEEP;

	public RegexProcessor() {
		// CommonToolsのバージョン確認
		boolean isValidVersion = CommonToolsVersionCheck
				.isValidVersion(requireCommonVersion);
		if (!isValidVersion) {
			int toolVersion = CommonToolsVersionCheck.getToolsVersion();
			String message = StringUtil.format(
					"CommonToolsのバージョンが古いためライブラリを使用できません:[要求Ver:{}][使用Ver:{}]",
					CommonToolsVersionCheck
							.getToolsVersionString(requireCommonVersion),
					CommonToolsVersionCheck.getToolsVersionString(toolVersion));
			throw new RuntimeException(message);
		}

		if (STATE.equals(RUN_STATE.SLEEP)) {
			PluginLoader loader = new PluginLoader();
			
			// 設定のロード
			loader.loadPlugin(JAutoRegexerSettings.class);

			// キャッシュプラグインのロード
			loader.loadPlugin(CachePlugin.class);

			// カスタムパーサプラグインのロード
			loader.loadPlugin(ParserPlugin.class);

			// プロセッサプラグインのロード
			loader.loadPlugin(ProcessorPlugin.class);
			


			// メソッドキャッシュオブジェクトを設定
			filterCache = CacheManager.getMethodCache();

			// フィールドキャッシュオブジェクトを設定
			fieldCache = CacheManager.getFieldCache();

			// 入力シーケンスプロセッサを設定
			processor = AutoInputSequenceProcessor.getInstance();

			STATE = RUN_STATE.RUNNING;
		}
	}

	/**
	 * 引数に指定された解析対象テキストを正規表現で解析し、第二引数に指定されたクラスオブジェクトのインスタンスに<br />
	 * 結果を格納します
	 * 
	 * @param <T>
	 * @param source
	 * @param cls
	 * @return
	 */
	public <T> List<T> process(final String source, final Class<T> cls) {
		List<T> resultList = new ArrayList<T>();
		if (source == null) {
			// logger.error("source is null");
			return resultList;
		}
		if (cls == null) {
			return resultList;
		}

		try {
			Pattern pattern = RegexPatternFactory.createPattern(cls);

			Matcher matcher = pattern.matcher(source);

			Map<Integer, List<Field>> fieldMap = fieldCache
					.getTargetFieldList(cls);

			
			while (matcher.find()) {
				T instance = cls.newInstance();
				// 現在の正規表現シーケンスにおいて
				// ループ
				for (Entry<Integer, List<Field>> entry : fieldMap.entrySet()) {
					List<Field> l = fieldMap.get(entry.getKey());
					for (Field targetField : l) {
						//System.out.println(matcher.groupCount());

						String inputSequence = matcher.group(entry.getKey());
						String mes = StringUtil.format("class[{}] source[{}] regex[{}] inputSequence[{}]"
								, cls.getCanonicalName()
								, source
								, pattern.toString()
								, inputSequence);
						//System.out.println(mes);
						if (processor.accept(targetField, inputSequence)) {
							processor.process(targetField, instance,
									inputSequence);
						}

					}
				}

				resultList.add(instance);
			}
			
			return resultList;
		} catch (PatternSyntaxException e) {
			// logger.error("pattern[" + regex.pattern()
			// + "] can't compile to java.util.regex.Pattern", e);
			return resultList;
		} catch (InstantiationException e) {
			e.printStackTrace();
			return resultList;
		} catch (IllegalAccessException e) {
			e.printStackTrace();
			return resultList;
		} catch (RegexPatternException e) {
			e.printStackTrace();
			return resultList;
		}

	}

	/**
	 * 第一引数で指定されたテキストコンテンツを第二引数で指定されたJavaBeans(RegexBean)で解析し
	 * 結果を第二引数で指定されたクラスオブジェクトの実態で返します。<BR />
	 * 
	 * このメソッドは、マッチした正規表現シーケンスを１回しか捜査しないことに注意してください。
	 * 即ち、複数のシーケンスに正規表現がヒットしても、その最初の要素しか捜査しません。
	 * このメソッドを使用することによって、文字列リストに結果をインジェクションする場合に マッチ件数が１件しか格納されません。
	 * 
	 * すべての正規表現シーケンスを全捜査したい場合は、processメソッドを利用してください。
	 * 
	 * @param <T>
	 *            RegexBean
	 * @param source
	 *            解析対象テキストコンテンツ
	 * @param cls
	 *            RegexBeanクラスオブジェクト
	 * @return 解析結果
	 */
	@SuppressWarnings("unchecked")
	public <T> T processBySingle(final String source, final Class<T> cls) {
		try {
			if (cls == null) {
				return null;
			}
			T result = cls.newInstance();
			if (source == null) {
				return result;
			}

			if (!cls.isAnnotationPresent(Regex.class)) {
				if (result.getClass().getCanonicalName()
						.equals("java.lang.String")) {
					result = (T) "";
					return result;
				} else {
					return result;
				}
			}

			Pattern pattern = RegexPatternFactory.createPattern(cls);

			Matcher matcher = pattern.matcher(source);

			Map<Integer, List<Field>> fieldMap = fieldCache
					.getTargetFieldList(cls);

			if (matcher.find()) {
				T instance = cls.newInstance();

				for (Entry<Integer, List<Field>> entry : fieldMap.entrySet()) {
					for (Field targetField : fieldMap.get(entry.getKey())) {
						String inputSequence = matcher.group(entry.getKey());
						if (processor.accept(targetField, inputSequence)) {
							processor.process(targetField, instance,
									inputSequence);
						}
					}
				}

				result = (instance);
			}

			return result;
		} catch (InstantiationException e) {
			e.printStackTrace();
			return null;
		} catch (IllegalAccessException e) {
			e.printStackTrace();

			return null;
		} catch (RegexPatternException e) {
			e.printStackTrace();
			return null;
		}
	}
}
