/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */

package jp.sourceforge.javacpt;

import java.lang.reflect.Method;
import java.util.logging.Logger;

import jp.sourceforge.javacpt.impl.ComparativePerformanceTestImpl;

/**
 * ComparativePerformanceTestHelper<br>
 * <br>
 * This helper makes you more DRY!<br>
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */

public class ComparativePerformanceTestHelper
{

	/**
	 * Create and initialize ComparativePerformanceTest instance.<br>
	 * 
	 * @param executeTimes
	 *            loop execute times
	 * @param instance
	 *            target instance(if invoke instance method)
	 * @return initialized test object
	 */
	public static ComparativePerformanceTest initialize(int executeTimes,
			Object instance)
	{
		ComparativePerformanceTest test = new ComparativePerformanceTestImpl();
		test.setExecuteTimes(executeTimes);
		test.setTargetInstance(instance);
		return test;
	}

	/**
	 * Create and initialize ComparativePerformanceTest instance with Logger.<br>
	 * 
	 * @param executeTimes
	 *            loop execute times
	 * @param instance
	 *            target instance(if invoke instance method)
	 * @param logger
	 *            Logger instance
	 * @return initialized test object
	 */
	public static ComparativePerformanceTest initialize(int executeTimes,
			Object instance, Logger logger)
	{
		ComparativePerformanceTest test = new ComparativePerformanceTestImpl();
		test.setExecuteTimes(executeTimes);
		test.setTargetInstance(instance);
		test.setLogger(logger);
		return test;
	}

	/**
	 * Invoke ComparativePerformanceTest.<br>
	 * (No parameter method)<br>
	 * 
	 * @param test
	 *            test object
	 * @param testName
	 *            test name
	 * @param methodName
	 *            test method name
	 * @return result millisec
	 * @throws NoSuchMethodException
	 */
	public static long invoke(ComparativePerformanceTest test, String testName,
			String methodName) throws NoSuchMethodException
	{
		return invoke(test, testName, methodName, (Object[]) null);
	}

	/**
	 * Invoke ComparativePerformanceTest.<br>
	 * (parameter types auto determinable)<br>
	 * 
	 * @param test
	 *            test object
	 * @param testName
	 *            test name
	 * @param method
	 *            method object
	 * @param methodParams
	 *            method param values
	 * @return result millisec
	 */
	public static long invoke(ComparativePerformanceTest test, String testName,
			Method method, Object... methodParams)
	{
		test.setTestName(testName);
		return test.execute(test.getExecuteTimes(), test.getTargetInstance(),
				method, (Object[]) methodParams);
	}

	/**
	 * Invoke ComparativePerformanceTest.<br>
	 * (array of parameter types and array of valued are given)<br>
	 * 
	 * @param test
	 *            test object
	 * @param testName
	 *            test name
	 * @param methodName
	 *            test method name
	 * @param methodParamTypes
	 *            method param types
	 * @param methodParams
	 *            method param values
	 * @return result millisec
	 * @throws NoSuchMethodException
	 */
	public static long invoke(ComparativePerformanceTest test, String testName,
			String methodName, Class<?>[] methodParamTypes,
			Object[] methodParams) throws NoSuchMethodException
	{
		Method method = test.getTargetInstance().getClass().getMethod(
				methodName, methodParamTypes);
		return invoke(test, testName, method, methodParams);
	}

	/**
	 * Invoke ComparativePerformanceTest.<br>
	 * (parameter type or value are given)<br>
	 * <br>
	 * It is interpreted as specified parameter type if first element was Class
	 * object.<br>
	 * <br>
	 * (ex.)<br>
	 * ComparativePerformanceTestHelper.invoke(test, testName, methodName,
	 * List.class, List.class, targetList1, targetList2);
	 * 
	 * @param test
	 *            test object
	 * @param testName
	 *            test name
	 * @param methodName
	 *            test method name
	 * @param methodParams
	 *            method param types or values
	 * @return result millisec
	 * @throws NoSuchMethodException
	 */
	public static long invoke(ComparativePerformanceTest test, String testName,
			String methodName, Object... methodParamTypesOrParamObjects)
			throws NoSuchMethodException
	{
		Class<?>[] methodParamTypes = null;
		if (methodParamTypesOrParamObjects != null)
		{
			int paramLen = methodParamTypesOrParamObjects.length;
			methodParamTypes = new Class[paramLen];
			boolean methodParamTypesAccepted = false;
			for (int i = 0; i < paramLen; i++)
			{
				if (methodParamTypesOrParamObjects[i] != null
						&& methodParamTypesOrParamObjects[i] instanceof Class[])
				{
					// re-create method param types
					methodParamTypes = new Class[1];
					methodParamTypes = (Class<?>[]) methodParamTypesOrParamObjects[i];
					// re-create method params
					Object[] methodParams = new Object[paramLen - 1];
					if (methodParamTypesOrParamObjects.length > 0)
					{
						for (int j = 1; j < paramLen; j++)
							methodParams[j - 1] = methodParamTypesOrParamObjects[j];
					}
					return invoke(test, testName, methodName, methodParamTypes,
							(Object[]) methodParams);

				} else if (methodParamTypesOrParamObjects[i] != null
						&& methodParamTypesOrParamObjects[i] instanceof Class)
				{
					methodParamTypes[i] = (Class<?>) methodParamTypesOrParamObjects[i];
					methodParamTypesAccepted = true;
				} else if (methodParamTypesAccepted)
				{
					// invalid pattern
					if (i != paramLen / 2)
						throw new IllegalArgumentException(
								"It is interpreted as specified param type"
										+ " if first element was Class object.");

					// re-create method param types
					int methodParamTypesLen = methodParamTypes.length;
					int methodParamTypesNewLen = methodParamTypesLen;
					for (int j = 0; j < methodParamTypesLen; j++)
					{
						if (methodParamTypes[j] == null)
						{
							methodParamTypesNewLen = j;
							break;
						}
					}
					Class<?>[] methodParamTypesTmp = new Class<?>[methodParamTypesNewLen];
					for (int j = 0; j < methodParamTypesNewLen; j++)
					{
						methodParamTypesTmp[j] = methodParamTypes[j];
					}
					methodParamTypes = methodParamTypesTmp;

					// re-create method params
					Object[] methodParams = new Object[paramLen - i];
					if (methodParamTypesOrParamObjects.length > 0)
					{
						for (int j = i; j < paramLen; j++)
							methodParams[j - i] = methodParamTypesOrParamObjects[j];
					}

					return invoke(test, testName, methodName, methodParamTypes,
							(Object[]) methodParams);
				} else
				{
					if (methodParamTypesOrParamObjects[i] == null)
						methodParamTypes[i] = Object.class;
					else
						methodParamTypes[i] = methodParamTypesOrParamObjects[i]
								.getClass();
				}
			}
		}
		Method method = test.getTargetInstance().getClass().getMethod(
				methodName, methodParamTypes);
		return invoke(test, testName, method, methodParamTypesOrParamObjects);
	}

}
