/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */

package jp.sourceforge.simplefh;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;

/**
 * SimpleFileReader<br>
 * <br>
 * Read file with simple methods.<br>
 * This module tries to read in the following procedure.<br>
 * <br>
 * 1. path on Jara resource.<br>
 * 2. full path on os file system.<br>
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */

public interface SimpleFileReader
{
	/**
	 * getDetectedEncoding
	 * 
	 * @param is
	 * @return
	 * @throws IOException
	 */
	public String getDetectedEncoding(InputStream is) throws IOException;

	/**
	 * getInputStreamFromResourceOrFileSystem
	 * 
	 * @param resourcePathOrFilePath
	 * @return Input Stream
	 * @throws FileNotFoundException
	 */
	public InputStream getInputStreamFromResourceOrFileSystem(
			String resourcePathOrFilePath) throws FileNotFoundException;

	/**
	 * Read file content(convert to char array)
	 * 
	 * @param resourcePathOrFilePath
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public char[] read2CharArray(String resourcePathOrFilePath) throws IOException;

	/**
	 * 
	 * Read file content(convert to char array)
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public char[] read2CharArray(File file) throws IOException;

	/**
	 * Read file content(convert to byte array)
	 * 
	 * @param resourcePathOrFilePath
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public byte[] read2ByteArray(String resourcePathOrFilePath) throws IOException;

	/**
	 * Read file content(convert to byte array)
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public byte[] read2ByteArray(File file) throws IOException;

	/**
	 * Read file content(convert to String List)
	 * 
	 * @param resourcePathOrFilePath
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public List<String> read2StringLineList(String resourcePathOrFilePath)
			throws IOException;

	/**
	 * 
	 * Read file content(convert to String List)
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public List<String> read2StringLineList(File file) throws IOException;

	/**
	 * Read file content(convert to String array)
	 * 
	 * @param resourcePathOrFilePath
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public String[] read2StringLineArray(String resourcePathOrFilePath)
			throws IOException;

	/**
	 * Read file content(convert to String array)
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public String[] read2StringLineArray(File file) throws IOException;

	public String getEncodingToRead();

	public void setEncodingToRead(String encodingToRead);

	public Integer getDefaultArrayLength();

	public void setDefaultArrayLength(Integer defaultArrayLength);
}
