/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */
package jp.sourceforge.javacpt.impl;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.logging.Logger;

import jp.sourceforge.javacpt.ComparativePerformanceTest;
import jp.sourceforge.javacpt.Printer;

/**
 * ComparativePerformanceTestImpl<br>
 * <br>
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */

public class ComparativePerformanceTestImpl implements
		ComparativePerformanceTest
{

	/**
	 * test name
	 */
	private String testName;

	/**
	 * start millisec
	 */
	private long startMillis;

	/**
	 * end millisec
	 */
	private long endMillis;

	/**
	 * print result flag
	 */
	private boolean printResultFlag = true;

	/**
	 * printer
	 */
	private Printer printer = null;

	/**
	 * default execute times
	 */
	public static final int DEFAULT_EXECUTE_TIMES = 1000;

	private int executeTimes = DEFAULT_EXECUTE_TIMES;

	private Object targetInstance = null;

	/**
	 * Default Constructor
	 */
	public ComparativePerformanceTestImpl()
	{
		this.printer = new PrinterImpl();
	}

	/**
	 * Constructor
	 * 
	 * @param testName
	 *            test name
	 */
	public ComparativePerformanceTestImpl(String testName)
	{
		this.testName = testName;
		this.printer = new PrinterImpl();
	}

	/**
	 * Record test start time(millisec)
	 */
	public void recordTestStart()
	{
		this.startMillis = System.currentTimeMillis();
	}

	/**
	 * Record test end time(millisec)
	 */
	public void recordTestEnd()
	{
		this.endMillis = System.currentTimeMillis();
	}

	/**
	 * Get result value
	 */
	public long getResult()
	{
		return this.endMillis - this.startMillis;
	}

	/**
	 * Set printResultFlag
	 */
	public void setPrintResultFlag(boolean printResultFlag)
	{
		this.printResultFlag = printResultFlag;
	}

	/**
	 * Print result value
	 * 
	 * @see PrinterImpl
	 */
	public void printResult()
	{
		if (this.printResultFlag)
		{
			String result = this.testName + " : "
					+ String.valueOf(this.endMillis - this.startMillis)
					+ "milsec.";
			printer.infoPrint(result);
		}
	}

	/**
	 * Print in ERROR level
	 * 
	 * @param str
	 *            String message to print
	 */
	public void errorPrint(String str)
	{
		printer.errorPrint(str);
	}

	/**
	 * Execute loop
	 * 
	 * @param executeTimes
	 *            loop execute times
	 * @param instance
	 *            target instance
	 * @param executeMethod
	 *            execute method object
	 * @param args
	 *            method parameters
	 */
	public void repeat(int executeTimes, Object instance, Method executeMethod,
			Object... args)
	{
		for (int i = 0; i < executeTimes; i++)
		{
			try
			{
				executeMethod.invoke(instance, args);
			} catch (IllegalArgumentException e)
			{
				String argsStr = "args : ";
				for (Object each : args)
				{
					argsStr += each.toString() + ",";
				}
				errorPrint("exception catched! : " + String.valueOf(i + 1)
						+ "/" + executeTimes + "(" + argsStr + ")");
				e.printStackTrace();
				break;
			} catch (IllegalAccessException e)
			{
				errorPrint("exception catched! : " + String.valueOf(i + 1)
						+ "/" + executeTimes);
				e.printStackTrace();
				break;
			} catch (InvocationTargetException e)
			{
				errorPrint("exception catched! : " + String.valueOf(i + 1)
						+ "/" + executeTimes);
				e.printStackTrace();
				break;
			}
		}
	}

	/**
	 * Execute loop test.
	 * 
	 * @param executeTimes
	 *            loop execute times
	 * @param instance
	 *            target instance(if execute instance method)
	 * @param executeMethod
	 *            execute method object
	 * @param args
	 *            method parameters
	 * @return result millisec
	 */
	public long execute(int executeTimes, Object instance,
			Method executeMethod, Object... args)
	{
		recordTestStart();
		repeat(executeTimes, instance, executeMethod, args);
		recordTestEnd();
		printResult();
		return getResult();
	}

	/**
	 * Execute loop test.
	 * 
	 * @param executeMethod
	 *            execute method object
	 * @param args
	 *            method parameters
	 * @return result millisec
	 */
	public long execute(Method execuMethod, Object... args)
	{
		return execute(this.executeTimes, this.targetInstance, execuMethod,
				args);
	}

	/**
	 * Get Logger
	 * 
	 * @return logger
	 */
	public Logger getLogger()
	{
		return printer.getLogger();
	}

	/**
	 * Set Logger
	 * 
	 * @param logger
	 */
	public void setLogger(Logger logger)
	{
		printer.setLogger(logger);
	}

	/**
	 * Get execute times
	 * 
	 * @return execute times
	 */
	public int getExecuteTimes()
	{
		return executeTimes;
	}

	/**
	 * Set execute times
	 * 
	 * @param executeTimes
	 *            execute times
	 */
	public void setExecuteTimes(int executeTimes)
	{
		this.executeTimes = executeTimes;
	}

	/**
	 * Get target instance
	 * 
	 * @return target instance
	 */
	public Object getTargetInstance()
	{
		return targetInstance;
	}

	/**
	 * Set target instance.
	 * 
	 * @param targetInstance
	 *            target instance
	 */
	public void setTargetInstance(Object targetInstance)
	{
		this.targetInstance = targetInstance;
	}

	/**
	 * Get test name.
	 * 
	 * @return test name
	 */
	public String getTestName()
	{
		return testName;
	}

	/**
	 * Set test name
	 * 
	 * @param testName
	 *            test name
	 */
	public void setTestName(String testName)
	{
		this.testName = testName;
	}

}
