/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */

package org.lightdi.container.util;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.lightdi.container.tag.LightDIConfigTag;

/**
 * LoggerUtil<br>
 * <br>
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */

public final class LightDILoggerUtil
{
	public static final String START = " START";
	public static final String END = " END";
	public static final String ERROR = " ERROR";

	private LightDILoggerUtil()
	{
	}

	public static final void info(String loggingType, Logger logger,
	        String containerName, String componentName, String msg, boolean logging)
	{
		if (logging)
			info(loggingType, logger, containerName, componentName, msg);
	}

	public static final void info(String loggingType, Logger logger,
	        String containerName, String componentName, String msg)
	{
		String prefix = getPrefixStringValue(loggingType, containerName, componentName);
		if (prefix == null)
			return;
		String sourceClass = new Throwable().getStackTrace()[1].getClassName();
		String sourceMethod = new Throwable().getStackTrace()[1].getMethodName();
		logger.logp(Level.INFO, sourceClass, sourceMethod, prefix + msg);
	}

	public static final void warn(String loggingType, Logger logger,
	        String containerName, String componentName, String msg, boolean logging)
	{
		if (logging)
			warn(loggingType, logger, containerName, componentName, msg);
	}

	public static final void warn(String loggingType, Logger logger,
	        String containerName, String componentName, String msg)
	{
		String prefix = getPrefixStringValue(loggingType, containerName, componentName);
		if (prefix == null)
			return;
		String sourceClass = new Throwable().getStackTrace()[1].getClassName();
		String sourceMethod = new Throwable().getStackTrace()[1].getMethodName();
		logger.logp(Level.WARNING, sourceClass, sourceMethod, prefix + msg);
	}

	public static final void fatal(String loggingType, Logger logger,
	        String containerName, String componentName, String msg, boolean logging)
	{
		if (logging)
			fatal(loggingType, logger, containerName, componentName, msg);
	}

	public static final void fatal(String loggingType, Logger logger,
	        String containerName, String componentName, String msg)
	{
		String prefix = getPrefixStringValue(loggingType, containerName, componentName);
		if (prefix == null)
			return;
		String sourceClass = new Throwable().getStackTrace()[1].getClassName();
		String sourceMethod = new Throwable().getStackTrace()[1].getMethodName();
		logger.logp(Level.SEVERE, sourceClass, sourceMethod, prefix + msg);
	}

	public static String getPrefixStringValue(String loggingType, String containerName,
	        String componentName)
	{
		String ret = "";
		if (containerName != null)
			ret += "[" + containerName + "]";
		if (componentName != null)
		{
			ret += "[" + componentName + "]";
			if (LightDIConfigTag.Attributes.LoadingLog.CONTAINERS_ONLY
			        .equals(loggingType))
				return null;
		}
		ret += " :: ";
		return ret;
	}

}
