/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */

package jp.sourceforge.simplefh.impl;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.simplefh.ClassPathResource;
import jp.sourceforge.simplefh.SimpleFileReader;
import jp.sourceforge.simplefh.constant.Encoding;
import jp.sourceforge.simplefh.util.FileResourceUtil;

/**
 * @see jp.sourceforge.simplefh.SimpleFileReader
 * @author Kazuhiro Sera
 * @version 1.0
 */

public class SimpleFileReaderImpl implements SimpleFileReader
{

	/**
	 * encoding to read file content
	 */
	private String encodingToRead = null;

	/**
	 * default array length
	 */
	private Integer defaultArrayLength = 4096;

	/*
	 * -------------------------------------------------------------------------
	 * char array
	 * -------------------------------------------------------------------------
	 */

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2CharArray(String)
	 */
	public char[] read2CharArray(String path) throws IOException
	{
		InputStream is = null;
		InputStreamReader isr = null;
		try
		{
			// get input stream
			is = FileResourceUtil.getInputStreamFromResourceOrFileSystem(path);
			// encoding detect
			if (this.encodingToRead == null)
			{
				// get input stream
				InputStream isTmp = this.getClass().getClassLoader().getResourceAsStream(
				        path);
				if (isTmp == null)
				{
					isTmp = new FileInputStream(path);
				}
				String detectedEncoding = FileResourceUtil.getDetectedEncoding(isTmp);
				this.encodingToRead = (detectedEncoding == null) ? Encoding.MS932
				        : detectedEncoding;
				FileResourceUtil.close(isTmp);
			}
			// get input stream reader
			isr = new InputStreamReader(is, this.encodingToRead);
			// substantially doubled if over default array length
			return _getFileContentInCharArray(isr, null, this.defaultArrayLength);
		} finally
		{
			FileResourceUtil.close(is);
			FileResourceUtil.close(isr);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2CharArray(File)
	 */
	public char[] read2CharArray(File file) throws IOException
	{
		InputStream is = null;
		InputStreamReader isr = null;
		try
		{
			// encoding detect
			if (this.encodingToRead == null)
			{
				// get input stream
				InputStream isTmp = new FileInputStream(file);
				String detectedEncoding = FileResourceUtil.getDetectedEncoding(isTmp);
				this.encodingToRead = (detectedEncoding == null) ? Encoding.MS932
				        : detectedEncoding;
				FileResourceUtil.close(isTmp);
			}
			// file system
			is = new FileInputStream(file);
			isr = new InputStreamReader(is, this.encodingToRead);
			// substantially doubled if over default array length
			return _getFileContentInCharArray(isr, null, this.defaultArrayLength);
		} finally
		{
			FileResourceUtil.close(isr);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2CharArray(jp.sourceforge
	 *      .simplefh.ClassPathResource)
	 */
	public char[] read2CharArray(ClassPathResource resourcePath) throws IOException
	{
		String path = resourcePath.getResourcePath();
		return read2CharArray(path);
	}

	/*
	 * -------------------------------------------------------------------------
	 * byte array
	 * -------------------------------------------------------------------------
	 */

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2ByteArray(String)
	 */
	public byte[] read2ByteArray(String path) throws IOException
	{
		InputStream is = null;
		try
		{
			// get input stream
			is = FileResourceUtil.getInputStreamFromResourceOrFileSystem(path);
			// encoding detect
			if (this.encodingToRead == null)
			{
				// get input stream
				InputStream isTmp = this.getClass().getClassLoader().getResourceAsStream(
				        path);
				if (isTmp == null)
				{
					isTmp = new FileInputStream(path);
				}
				String detectedEncoding = FileResourceUtil.getDetectedEncoding(isTmp);
				this.encodingToRead = (detectedEncoding == null) ? Encoding.MS932
				        : detectedEncoding;

				FileResourceUtil.close(isTmp);
			}
			// substantially doubled if over default array length
			byte[] retArr = _getFileContentInByteArray(is, null, this.defaultArrayLength);
			return new String(retArr, this.encodingToRead).getBytes(Encoding.MS932);
		} finally
		{
			FileResourceUtil.close(is);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2ByteArray(File)
	 */
	public byte[] read2ByteArray(File file) throws IOException
	{
		BufferedInputStream bis = null;
		byte[] retByteArr = null;
		try
		{
			// encoding detect
			if (this.encodingToRead == null)
			{
				// get input stream
				InputStream isTmp = new FileInputStream(file);
				String detectedEncoding = FileResourceUtil.getDetectedEncoding(isTmp);
				this.encodingToRead = (detectedEncoding == null) ? Encoding.MS932
				        : detectedEncoding;
				FileResourceUtil.close(isTmp);
			}
			bis = new BufferedInputStream(new FileInputStream(file));
			retByteArr = new byte[(int) file.length()];
			bis.read(retByteArr);
		} finally
		{
			FileResourceUtil.close(bis);
		}
		return new String(retByteArr, this.encodingToRead).getBytes(Encoding.MS932);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2ByteArray(jp.sourceforge
	 *      .simplefh.ClassPathResource)
	 */
	public byte[] read2ByteArray(ClassPathResource resourcePath) throws IOException
	{
		String path = resourcePath.getResourcePath();
		return read2ByteArray(path);
	}

	/*
	 * -------------------------------------------------------------------------
	 * String List
	 * -------------------------------------------------------------------------
	 */

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2StringLineList(String)
	 */
	public List<String> read2StringLineList(String path) throws IOException
	{
		List<String> retList = new ArrayList<String>();
		InputStream is = null;
		InputStreamReader isr = null;
		BufferedReader br = null;
		try
		{
			// get input stream
			is = FileResourceUtil.getInputStreamFromResourceOrFileSystem(path);
			// encoding detect
			if (this.encodingToRead == null)
			{
				// get input stream
				InputStream isTmp = this.getClass().getClassLoader().getResourceAsStream(
				        path);
				if (isTmp == null)
				{
					isTmp = new FileInputStream(path);
				}
				String detectedEncoding = FileResourceUtil.getDetectedEncoding(isTmp);
				this.encodingToRead = (detectedEncoding == null) ? Encoding.MS932
				        : detectedEncoding;
				FileResourceUtil.close(isTmp);
			}
			isr = new InputStreamReader(is, this.encodingToRead);
			br = new BufferedReader(isr);
			String line = null;
			while ((line = br.readLine()) != null)
			{
				retList.add(line);
			}
		} finally
		{
			FileResourceUtil.close(is);
			FileResourceUtil.close(isr);
			FileResourceUtil.close(br);
		}
		return retList;
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2StringLineList(File)
	 */
	public List<String> read2StringLineList(File file) throws IOException
	{
		List<String> retList = new ArrayList<String>();
		InputStream is = null;
		InputStreamReader isr = null;
		BufferedReader br = null;
		try
		{
			// encoding detect
			if (this.encodingToRead == null)
			{
				// get input stream
				InputStream isTmp = new FileInputStream(file);
				String detectedEncoding = FileResourceUtil.getDetectedEncoding(isTmp);
				this.encodingToRead = (detectedEncoding == null) ? Encoding.MS932
				        : detectedEncoding;
				FileResourceUtil.close(isTmp);
			}
			// file system
			is = new FileInputStream(file);
			isr = new InputStreamReader(is, this.encodingToRead);
			br = new BufferedReader(isr);
			String line = null;
			while ((line = br.readLine()) != null)
			{
				retList.add(line);
			}
		} finally
		{
			FileResourceUtil.close(is);
			FileResourceUtil.close(isr);
			FileResourceUtil.close(br);
		}
		return retList;
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2StringLineList(jp.sourceforge
	 *      .simplefh.ClassPathResource)
	 */
	public List<String> read2StringLineList(ClassPathResource resourcePath)
	        throws IOException
	{
		String path = resourcePath.getResourcePath();
		return read2StringLineList(path);
	}

	/*
	 * -------------------------------------------------------------------------
	 * String array
	 * -------------------------------------------------------------------------
	 */

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2StringLineArray(String)
	 */
	public String[] read2StringLineArray(String path) throws IOException
	{
		List<String> strList = this.read2StringLineList(path);
		String[] retArr = new String[strList.size()];
		int retArrLen = retArr.length;
		for (int i = 0; i < retArrLen; i++)
		{
			retArr[i] = strList.get(i);
		}
		return retArr;
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2StringLineArray(File)
	 */
	public String[] read2StringLineArray(File file) throws IOException
	{
		List<String> strList = this.read2StringLineList(file);
		String[] retArr = new String[strList.size()];
		int retArrLen = retArr.length;
		for (int i = 0; i < retArrLen; i++)
		{
			retArr[i] = strList.get(i);
		}
		return retArr;
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileReader#read2StringLineArray(jp.sourceforge
	 *      .simplefh.ClassPathResource)
	 */
	public String[] read2StringLineArray(ClassPathResource resourcePath)
	        throws IOException
	{
		String path = resourcePath.getResourcePath();
		return read2StringLineArray(path);
	}

	/*
	 * -------------------------------------------------------------------------
	 * Private Implementation
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Get byte array(file content) in necessary and sufficient array length
	 * 
	 * @param isr
	 *            InputStreamReader obj
	 * @param lastArr
	 *            arg char array from invoker(recursive execute)
	 * @return return char array
	 * @throws IOException
	 *             file read error
	 */
	private byte[] _getFileContentInByteArray(InputStream is, byte[] registeredArr,
	        int bufArrLen) throws IOException
	{
		int registeredArrLen = (registeredArr == null) ? 0 : registeredArr.length;
		// buffering array to read
		byte[] bufArr = new byte[bufArrLen];
		is.read(bufArr);
		Integer endIdxOfArr = null;
		// check end index of array
		for (int i = 0; i < bufArrLen; i++)
		{
			if (bufArr[i] == 0)
			{
				endIdxOfArr = i;
				break;
			}
		}
		byte[] resultArr = null;
		// merge registered array and buffering array
		if (endIdxOfArr == null)
		{
			// not end index of array
			if (registeredArrLen == 0)
			{
				resultArr = new byte[bufArrLen];
				for (int i = 0; i < bufArrLen; i++)
				{
					resultArr[i] = bufArr[i];
				}
			} else
			{
				resultArr = new byte[registeredArrLen + bufArrLen];
				System.arraycopy(registeredArr, 0, resultArr, 0, registeredArrLen);
				System.arraycopy(bufArr, 0, resultArr, registeredArrLen, bufArrLen);
			}
			// recursive execute
			return _getFileContentInByteArray(is, bufArr, bufArrLen);
		} else
		{
			// end index of array
			if (registeredArrLen == 0)
			{
				resultArr = new byte[endIdxOfArr];
				for (int i = 0; i < endIdxOfArr; i++)
				{
					resultArr[i] = bufArr[i];
				}
			} else
			{
				resultArr = new byte[registeredArrLen + endIdxOfArr];
				System.arraycopy(registeredArr, 0, resultArr, 0, registeredArrLen);
				System.arraycopy(bufArr, 0, resultArr, registeredArrLen, endIdxOfArr);
			}
		}
		return resultArr;
	}

	/**
	 * Get char array(file content) in necessary and sufficient array length
	 * 
	 * @param isr
	 *            InputStreamReader obj
	 * @param registeredArr
	 *            already regsitered char array (array from recursive invoker)
	 * @return return char array
	 * @throws IOException
	 *             file read error
	 */
	private char[] _getFileContentInCharArray(InputStreamReader isr,
	        char[] registeredArr, int bufArrLen) throws IOException
	{
		int registeredArrLen = (registeredArr == null) ? 0 : registeredArr.length;
		// buffering array to read
		char[] bufArr = new char[bufArrLen];
		isr.read(bufArr);
		Integer endIdxOfArr = null;
		// check end index of array
		for (int i = 0; i < bufArrLen; i++)
		{
			if (bufArr[i] == 0)
			{
				endIdxOfArr = i;
				break;
			}
		}
		char[] resultArr = null;
		// merge registered array and buffering array
		if (endIdxOfArr == null)
		{
			// not end index of array
			if (registeredArrLen == 0)
			{
				resultArr = new char[bufArrLen];
				for (int i = 0; i < bufArrLen; i++)
				{
					resultArr[i] = bufArr[i];
				}
			} else
			{
				resultArr = new char[registeredArrLen + bufArrLen];
				System.arraycopy(registeredArr, 0, resultArr, 0, registeredArrLen);
				System.arraycopy(bufArr, 0, resultArr, registeredArrLen, bufArrLen);
			}
			// recursive execute
			return _getFileContentInCharArray(isr, bufArr, bufArrLen);
		} else
		{
			// end index of array
			if (registeredArrLen == 0)
			{
				resultArr = new char[endIdxOfArr];
				for (int i = 0; i < endIdxOfArr; i++)
				{
					resultArr[i] = bufArr[i];
				}
			} else
			{
				resultArr = new char[registeredArrLen + endIdxOfArr];
				System.arraycopy(registeredArr, 0, resultArr, 0, registeredArrLen);
				System.arraycopy(bufArr, 0, resultArr, registeredArrLen, endIdxOfArr);
			}
		}
		return resultArr;
	}

	/*
	 * -------------------------------------------------------------------------
	 * Getter/Setter
	 * -------------------------------------------------------------------------
	 */

	public String getEncodingToRead()
	{
		return encodingToRead;
	}

	public SimpleFileReader setEncodingToRead(String encodingToRead)
	{
		this.encodingToRead = encodingToRead;
		return this;
	}

	public Integer getDefaultArrayLength()
	{
		return defaultArrayLength;
	}

	public SimpleFileReader setDefaultArrayLength(Integer defaultArrayLength)
	{
		this.defaultArrayLength = defaultArrayLength;
		return this;
	}

}
