/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */

package jp.sourceforge.simplefh.impl;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.List;

import jp.sourceforge.simplefh.ClassPathResource;
import jp.sourceforge.simplefh.SimpleFileWriter;
import jp.sourceforge.simplefh.constant.BR;
import jp.sourceforge.simplefh.constant.Encoding;
import jp.sourceforge.simplefh.util.FileResourceUtil;

/**
 * @see jp.sourceforge.simplefh.SimpleFileWriter
 * @author Kazuhiro Sera
 * @version 1.0
 */

public class SimpleFileWriterImpl implements SimpleFileWriter
{
	private String encoding = Encoding.MS932;
	private String writeTargetAbsolutePath = null;
	private String lineBreak = BR.CRLF;

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#write(String[])
	 */
	public void write(String[] lines) throws IOException
	{
		StringBuffer sb = new StringBuffer();
		int len = lines.length;
		for (int i = 0; i < len; i++)
		{
			sb.append(lines[i]);
			sb.append(this.lineBreak);
		}
		String strContent = sb.toString();
		write(strContent);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#write(List)
	 */
	public void write(List<String> lines) throws IOException
	{
		StringBuffer sb = new StringBuffer();
		int len = lines.size();
		for (int i = 0; i < len; i++)
		{
			sb.append(lines.get(i));
			sb.append(this.lineBreak);
		}
		String strContent = sb.toString();
		write(strContent);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#write(String)
	 */
	public void write(String strContent) throws IOException
	{
		byte[] content = strContent.getBytes(this.encoding);
		write(content);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#write(byte[])
	 */
	public void write(byte[] content) throws IOException
	{
		FileOutputStream fos = null;
		try
		{
			// cannot write if write target file is included in jar file
			if (!FileResourceUtil.isFileAccessAvailable(writeTargetAbsolutePath))
			{
				throw new IOException("Cannot write to resource file in archive file : "
				        + writeTargetAbsolutePath);
			}
			// write execute
			fos = new FileOutputStream(writeTargetAbsolutePath);
			fos.write(content);
		} catch (NullPointerException npe)
		{
			// no target file found
			throw new IllegalStateException("The target file to write is required.");
		} catch (IOException e)
		{
			// ex. file locked
			throw new IOException("Cannot write content to the target file : "
			        + writeTargetAbsolutePath);
		} finally
		{
			FileResourceUtil.close(fos);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#getLineBreak()
	 */
	public String getLineBreak()
	{
		return lineBreak;
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#setLineBreak(java.lang.String)
	 */
	public SimpleFileWriter setLineBreak(String lineBreak)
	{
		this.lineBreak = lineBreak;
		return this;
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#getEncoding()
	 */
	public String getEncoding()
	{
		return encoding;
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#setEncoding(java.lang.String)
	 */
	public SimpleFileWriter setEncoding(String encoding)
	{
		this.encoding = encoding;
		return this;
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#getWriteTarget()
	 */
	public String getWriteTarget()
	{
		return writeTargetAbsolutePath;
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#setWriteTarget(java.lang.String)
	 */
	public SimpleFileWriter setWriteTarget(String path)
	{
		writeTargetAbsolutePath = path;
		return this;
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#setWriteTarget(jp.sourceforge
	 *      .simplefh.ClassPathResource)
	 */
	public SimpleFileWriter setWriteTarget(ClassPathResource resourcePath)
	{
		String absolutePath = FileResourceUtil.getFileSystemAbsolutePath(resourcePath
		        .getResourcePath());
		return setWriteTarget(absolutePath);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileWriter#setWriteTarget(java.io.File)
	 */
	public SimpleFileWriter setWriteTarget(File file)
	{
		String absolutePath = file.getAbsolutePath();
		return setWriteTarget(absolutePath);
	}

}
