/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */

package jp.sourceforge.simplefh.util;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;

import jp.sourceforge.simplefh.constant.FixedValue;

import org.mozilla.universalchardet.UniversalDetector;

/**
 * Utility for file resource.
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */

public class FileResourceUtil
{

	/**
	 * Judge target absolute path is available to access on os file system.
	 * 
	 * @param absolutePath
	 *            target file absolute path
	 * @return result boolean value
	 */
	public static boolean isFileAccessAvailable(String absolutePath)
	{
		absolutePath = getFileSystemAbsolutePath(absolutePath);
		if (absolutePath.matches("^file\\:.*"))
		{
			return true;
		} else if (absolutePath.matches("^jar\\:.*"))
		{
			return false;
		} else
		{
			// TODO maybe true
			return true;
		}
	}

	/**
	 * Get detected encoding.
	 * 
	 * @param is
	 * @return
	 * @throws IOException
	 */
	public static String getDetectedEncoding(InputStream is) throws IOException
	{
		UniversalDetector detector = new UniversalDetector(null);

		byte[] buf = new byte[4096];
		int nread;
		while ((nread = is.read(buf)) > 0 && !detector.isDone())
		{
			detector.handleData(buf, 0, nread);
		}
		detector.dataEnd();
		return detector.getDetectedCharset();
	}

	/**
	 * Replace all back slash to slash.<br>
	 * (for convert Windows platform directory separator)<br>
	 * 
	 * @param target
	 *            target absolute file path
	 * @return result
	 */
	public static String replaceAllBackSlash2Slash(String absolutePath)
	{
		return absolutePath.replaceAll(FixedValue.BACK_SLASH_ESCAPED, FixedValue.SLASH);
	}

	/**
	 * Get file system absolute path.
	 * 
	 * @param resourcePathOrFilePath
	 *            target path(fs path or resource path)
	 * @return fs absolute path
	 */
	public static String getFileSystemAbsolutePath(String resourcePathOrFilePath)
	{

		resourcePathOrFilePath = resourcePathOrFilePath.replaceAll(
		        FixedValue.BACK_SLASH_ESCAPED, FixedValue.SLASH);
		String absolutePath = null;
		URL url = FileResourceUtil.class.getClassLoader().getResource(
		        resourcePathOrFilePath);
		if (url == null)
		{
			// new file create pattern
			String parentDir = FixedValue.EMPTY;
			String[] pathArr = resourcePathOrFilePath.split(FixedValue.SLASH);
			int pathArrLen = pathArr.length;
			for (int i = 0; i < pathArrLen - 1; i++)
				parentDir += pathArr[i] + FixedValue.SLASH;
			URL parentUrl = FileResourceUtil.class.getClassLoader()
			        .getResource(parentDir);
			if (parentUrl != null)
				absolutePath = parentUrl.toString().replaceFirst("file:/",
				        FixedValue.EMPTY)
				        + pathArr[pathArrLen - 1];
		}
		if (absolutePath == null)
		{
			if (url != null)
				// get absolute path from resource path
				absolutePath = url.toString().replaceFirst("file:/", FixedValue.EMPTY);
			else
				// absolute path or file does not exist
				absolutePath = resourcePathOrFilePath;
		}
		return absolutePath;
	}

	/**
	 * Get Resource root path.
	 * 
	 * @param resourcePath
	 *            resource path
	 * @param absolutePath
	 *            file system absolute path
	 * @return resource root path
	 */
	public static String getResourceRootPath(String resourcePath, String absolutePath)
	{
		resourcePath = resourcePath.replaceAll(FixedValue.BACK_SLASH_ESCAPED,
		        FixedValue.SLASH);
		String[] resArr = resourcePath.split(FixedValue.SLASH);
		absolutePath = absolutePath.replace(FixedValue.BACK_SLASH_ESCAPED,
		        FixedValue.SLASH);
		String[] abArr = absolutePath.split(FixedValue.SLASH);
		int abLen = abArr.length;
		int offset = 0;
		int rootPathIndex = 0;
		try
		{
			for (int i = 0; i < abLen; i++)
			{
				if (offset == 0 && abArr[i].equals(resArr[offset]))
				{
					rootPathIndex = i - 1;
					offset++;
				} else if (abArr[i].equals(resArr[offset]))
				{
					offset++;
				}
			}
		} catch (Exception e)
		{
			throw new IllegalArgumentException(
			        "Cannot get resource root path (resource path :" + resourcePath
			                + ", absolute path : " + absolutePath + ")");
		}
		if (offset != resArr.length)
			throw new IllegalArgumentException(
			        "Cannot get resource root path (resource path :" + resourcePath
			                + ", absolute path : " + absolutePath + ")");

		String retStr = "";
		for (int i = 0; i <= rootPathIndex; i++)
			retStr += abArr[i] + FixedValue.SLASH;
		return retStr;
	}

	/**
	 * Get InputStream from resource path or file system full path.
	 * 
	 * @param resourcePathOrFilePath
	 * @return Input Stream
	 * @throws FileNotFoundException
	 */
	public static InputStream getInputStreamFromResourceOrFileSystem(
	        String resourcePathOrFilePath) throws FileNotFoundException
	{
		// replace delimiter class path resource on windows
		resourcePathOrFilePath = FileResourceUtil
		        .replaceAllBackSlash2Slash(resourcePathOrFilePath);
		// try to get resource file stream
		InputStream is = FileResourceUtil.class.getClassLoader().getResourceAsStream(
		        resourcePathOrFilePath);
		if (is == null)
		{
			// cannot get stream from resource, and try to access file system
			is = new FileInputStream(resourcePathOrFilePath);
		}
		return is;

	}

	/**
	 * Close resource safely.<br>
	 * 
	 * @param is
	 *            InputStream resource object
	 */
	public static void close(InputStream is)
	{
		try
		{
			if (is != null)
				is.close();
		} catch (Exception ignore)
		{
		}
	}

	/**
	 * Close resource safely.<br>
	 * 
	 * @param isr
	 *            InputStreamReader resource object
	 */
	public static void close(InputStreamReader isr)
	{
		try
		{
			if (isr != null)
				isr.close();
		} catch (Exception ignore)
		{
		}
	}

	/**
	 * Close resource safely.<br>
	 * 
	 * @param br
	 *            BufferedReader resource object
	 */
	public static void close(BufferedReader br)
	{
		try
		{
			if (br != null)
				br.close();
		} catch (Exception ignore)
		{
		}
	}

	/**
	 * Close resource safely.<br>
	 * 
	 * @param bis
	 *            BufferedInputStream resource object
	 */
	public static void close(BufferedInputStream bis)
	{
		try
		{
			if (bis != null)
				bis.close();
		} catch (Exception ignore)
		{
		}
	}

	/**
	 * Close resource safely.<br>
	 * 
	 * @param bis
	 *            FileOutputStream resource object
	 */
	public static void close(FileOutputStream fos)
	{
		try
		{
			if (fos != null)
				fos.close();
		} catch (Exception ignore)
		{
		}
	}

}
