/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views;

import java.io.File;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.util.LocalSelectionTransfer;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.TreeColumnViewerLabelProvider;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.DropTargetAdapter;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.DropTargetListener;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.part.ViewPart;

import ch.kuramo.javie.app.ColorUtil;
import ch.kuramo.javie.app.DropTargetEnhancement;
import ch.kuramo.javie.app.UIUtil;
import ch.kuramo.javie.app.actions.NewCompositionAction;
import ch.kuramo.javie.app.actions.NewFileItemsAction;
import ch.kuramo.javie.app.actions.NewFolderAction;
import ch.kuramo.javie.app.actions.NewSolidColorItemAction;
import ch.kuramo.javie.app.actions.RenameItemAction;
import ch.kuramo.javie.app.project.NewFileItemsOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.RenameItemOperation;
import ch.kuramo.javie.app.project.ReparentItemsOperation;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.Util;

public class ProjectView extends ViewPart {

	public static final String ID = "ch.kuramo.javie.app.views.projectView";

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");


	private TreeViewer _treeViewer;

	private boolean _editName;


	public void createPartControl(Composite parent) {
		ProjectManager pm = ProjectManager.forWorkbenchWindow(getSite().getWorkbenchWindow());
		if (pm == null) {
			throw new IllegalStateException("No ProjectManager exists for this window.");
		}

		// TODO サムネールと詳細情報を表示する領域を作る。

		_treeViewer = new TreeViewer(parent, SWT.MULTI);
		_treeViewer.setContentProvider(new ItemTreeContentProvider());

		Tree tree = _treeViewer.getTree();
		tree.setHeaderVisible(true);
		tree.setLinesVisible(!COCOA);

		tree.setBackground(ColorUtil.tableBackground());

		if (COCOA) {
			final Color lineColor = ColorUtil.tableRowLine();

			tree.addListener(SWT.EraseItem, new Listener() {
				public void handleEvent(Event event) {
					int x = event.x;
					int y = event.y + event.height - 1;
					GC gc = event.gc;
					gc.setForeground(lineColor);
					gc.drawLine(x, y, x + event.width, y);
				}
			});
		}

		TreeViewerColumn nameViewerColumn = createViewerColumn(200, null, "名前");
		setEditingSupportForName(nameViewerColumn);

		// TODO ラベルのアイコン
		Image labelIcon = null;
		createViewerColumn( 25, labelIcon, "");

		createViewerColumn( 90, null, "種類");
		createViewerColumn( 60, null, "サイズ");
		createViewerColumn( 90, null, "ﾃﾞｭﾚｰｼｮﾝ");
		createViewerColumn(300, null, "ファイルパス");

		initContextMenu();
		initDragAndDrop();
		initDoubleClick();

		_treeViewer.setInput(pm);
	}

	private TreeViewerColumn createViewerColumn(int width, Image image, String name) {
		TreeViewerColumn viewerColumn = new TreeViewerColumn(_treeViewer, SWT.NONE);
		TreeColumn column = viewerColumn.getColumn();
		column.setWidth(width);
		column.setImage(image);
		column.setText(name);
		column.setMoveable(true);

		viewerColumn.setLabelProvider(new TreeColumnViewerLabelProvider(new ItemLabelProvider()));

		return viewerColumn;
	}

	private void setEditingSupportForName(TreeViewerColumn viewerColumn) {
		viewerColumn.setEditingSupport(new EditingSupport(_treeViewer) {
			private final TextCellEditor cellEditor;

			{
				cellEditor = new TextCellEditor(_treeViewer.getTree(), SWT.SINGLE | SWT.BORDER);
				Control control = cellEditor.getControl();
				control.setBackground(control.getDisplay().getSystemColor(SWT.COLOR_WHITE));
			}

			protected boolean canEdit(Object element) {
				return _editName;
			}

			protected CellEditor getCellEditor(Object element) {
				return cellEditor;
			}

			protected Object getValue(Object element) {
				return ((Item) element).getName();
			}

			protected void setValue(Object element, Object value) {
				ProjectManager pm = (ProjectManager) _treeViewer.getInput();
				pm.postOperation(new RenameItemOperation(pm, (Item) element, String.valueOf(value)));
			}
		});
	}

	private void initContextMenu() {
		IWorkbenchWindow window = getSite().getWorkbenchWindow();

		MenuManager menuMgr = new MenuManager();
		MenuManager inputMenu = new MenuManager("読み込み");

		menuMgr.add(new NewCompositionAction(window));
		menuMgr.add(new NewSolidColorItemAction(window));
		menuMgr.add(new NewFolderAction(window));
		menuMgr.add(inputMenu);
		menuMgr.add(new Separator());
		menuMgr.add(new RenameItemAction(window, _treeViewer));
		menuMgr.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));

		inputMenu.add(new NewFileItemsAction(window));

		getSite().registerContextMenu(menuMgr, _treeViewer);

		Control treeViewControl = _treeViewer.getControl();
		Menu treeViewMenu = menuMgr.createContextMenu(treeViewControl);
		treeViewControl.setMenu(treeViewMenu);
	}

	private void initDragAndDrop() {
		final LocalSelectionTransfer selectionTransfer = LocalSelectionTransfer.getTransfer();

		DragSourceListener sourceListener = new DragSourceListener() {
			public void dragStart(DragSourceEvent event) {
				ISelection selection = new LocalSelectionWrapper(_treeViewer);
				if (selection.isEmpty()) {
					event.doit = false;
				} else {
					selectionTransfer.setSelection(selection);
					selectionTransfer.setSelectionSetTime(event.time & 0xFFFFFFFFL);
					event.doit = true;
				}
			}

			public void dragSetData(DragSourceEvent event) {
				event.data = selectionTransfer.getSelection();
			}

			public void dragFinished(DragSourceEvent event) {
				selectionTransfer.setSelection(null);
				selectionTransfer.setSelectionSetTime(0);
			}
		};

		final FileTransfer fileTransfer = FileTransfer.getInstance();

		DropTargetListener targetListener = new DropTargetAdapter() {
			private Item getDestinationItem(TreeItem treeItem, boolean compAllowed) {
				if (treeItem == null) {
					return null;
				}
				Object data = treeItem.getData();
				if (!(data instanceof Folder || (compAllowed && data instanceof CompositionItem))) {
					treeItem = treeItem.getParentItem();
					data = (treeItem != null) ? treeItem.getData() : null;
				}
				return (Item) data;
			}

			private boolean isAcceptable(DropTargetEvent event) {
				if (fileTransfer.isSupportedType(event.currentDataType)) {
					return true;
				}

				if (!selectionTransfer.isSupportedType(event.currentDataType)) {
					return false;
				}

				ITreeSelection selection = LocalSelectionWrapper.getSelection(selectionTransfer, _treeViewer);
				if (selection == null) {
					return false;
				}

				Item destItem = getDestinationItem((TreeItem) event.item, true);
				if (destItem == null) {
					return true;
				}

				// 自分の子にドロップしようとしている場合のチェック
				Set<Folder> selectedFolders = Util.newSet();
				for (Object o : selection.toList()) {
					if (o instanceof Folder) {
						selectedFolders.add((Folder) o);
					}
				}
				for (Item i = destItem; i != null; i = i.getParent()) {
					if (selectedFolders.contains(i)) {
						return false;
					}
				}

				return true;
			}

			public void dragOver(DropTargetEvent event) {
				if (!isAcceptable(event)) {
					event.detail = DND.DROP_NONE;
					event.feedback = DND.FEEDBACK_NONE;
					return;
				}

				int detail = event.operations & DND.DROP_MOVE;

				if (event.item != null) {
					boolean isFile = fileTransfer.isSupportedType(event.currentDataType);
					Item destItem = getDestinationItem((TreeItem) event.item, !isFile);
					if (destItem != event.item.getData()) {
						// フィードバックがフォルダ上で起こるようにする。
						event.feedback = (event.feedback | DND.FEEDBACK_SELECT | DropTargetEnhancement.FEEDBACK_PARENT)
									& ~DND.FEEDBACK_INSERT_AFTER & ~DND.FEEDBACK_INSERT_BEFORE;

					} else if (destItem instanceof CompositionItem) {
						detail = event.operations & DND.DROP_LINK;
					}
				}

				event.detail = detail;
			}

			public void dropAccept(DropTargetEvent event) {
				dragOver(event);
			}

			public void drop(DropTargetEvent event) {
				boolean isFile = fileTransfer.isSupportedType(event.currentDataType);
				Item destItem = getDestinationItem((TreeItem) event.item, !isFile);

				ProjectManager pm = (ProjectManager) _treeViewer.getInput();

				if (isFile) {
					Set<File> files = Util.newSet();
					for (String path : (String[]) event.data) {
						files.add(new File(path));
					}
					pm.postOperation(new NewFileItemsOperation(pm, (Folder) destItem, files));

				} else {
					ITreeSelection selection = LocalSelectionWrapper.getSelection(selectionTransfer, _treeViewer);
					if (selection == null) {
						event.detail = DND.DROP_NONE;
						return;
					}

					if (destItem instanceof CompositionItem) {
						// TODO
					} else {
						@SuppressWarnings("unchecked")
						List<Item> items = selection.toList();
						pm.postOperation(new ReparentItemsOperation(pm, items, (Folder) destItem));
					}
				}
			}
		};

		_treeViewer.addDragSupport(DND.DROP_MOVE | DND.DROP_LINK, new Transfer[] { selectionTransfer }, sourceListener);
		_treeViewer.addDropSupport(DND.DROP_MOVE | DND.DROP_LINK, new Transfer[] { selectionTransfer, fileTransfer }, targetListener);
	}

	private void initDoubleClick() {
		class DoubleClickListener extends MouseAdapter implements IDoubleClickListener {
			private int modifiers;

			public void mouseDown(MouseEvent event) {
				modifiers = event.stateMask & SWT.MODIFIER_MASK;
			}

			public void doubleClick(DoubleClickEvent event) {
				if (modifiers != 0) {
					return;
				}

				ITreeSelection selection = (ITreeSelection) _treeViewer.getSelection();

				// こっちだとなぜか、ダブルクリックのときに選択項目をひとつしか返さない（Enterキーなら複数返す）
				//ITreeSelection selection = (ITreeSelection) event.getSelection();	

				for (Iterator<?> it = selection.iterator(); it.hasNext(); ) {
					Object item = it.next();
					if (item instanceof Item) {
						openItem((Item) item);
					}
				}
			}
		}

		DoubleClickListener listener = new DoubleClickListener();
		_treeViewer.getTree().addMouseListener(listener);
		_treeViewer.addDoubleClickListener(listener);
	}

	public void setFocus() {
		_treeViewer.getControl().setFocus();
	}

	public Folder getItemTreeInsertionPoint() {
		TreeSelection selection = (TreeSelection) _treeViewer.getSelection();
		if (selection.isEmpty()) {
			return null;
		}

		TreePath[] pathes = selection.getPaths();
		if (pathes.length == 1) {
			Item item = (Item) pathes[0].getLastSegment();
			return (item instanceof Folder) ? (Folder) item : item.getParent();
		}

		// 以下は AE CS3 とたぶん同じ動作。そこまでする必要も無いけど。

		Folder folder = null;
		for (int i = 1; ; ++i) {
			Folder tmp = null;
			for (TreePath path : pathes) {
				if (path.getSegmentCount() <= i) {
					return folder;
				}
				Folder parent = ((Item) path.getSegment(i)).getParent();
				if (tmp == null) {
					tmp = parent;
				} else if (tmp != parent) {
					return folder;
				}
			}
			folder = tmp;
		}
	}

	public void editName(Item item) {
		_editName = true;
		_treeViewer.editElement(item, 0);
		_editName = false;
	}

	private void openItem(Item item) {
		IWorkbenchPage page = getSite().getPage();

		if (item instanceof CompositionItem) {
			if (((CompositionItem) item).getComposition() instanceof LayerComposition) {
				UIUtil.showView(page, LayerCompositionView.ID, item.getId(), IWorkbenchPage.VIEW_ACTIVATE);
				UIUtil.showView(page, MediaPlayerView.ID, item.getId(), IWorkbenchPage.VIEW_VISIBLE);
				return;
			}
		}

		if (item instanceof MediaItem) {
			UIUtil.showView(page, MediaPlayerView.ID, item.getId(), IWorkbenchPage.VIEW_ACTIVATE);
		}
	}

}
