/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.util.List;
import java.util.Map;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.TreeItem;

import ch.kuramo.javie.app.ImageUtil;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.ProjectOperation;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.ArithmeticalAnimatableValue;
import ch.kuramo.javie.core.Util;

public abstract class AnimatableValueElementDelegate<V> {

	private static final int[] UNDERLINE_DASH = new int[] { 2, 2 };

	protected static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	protected static final boolean WIN32 = SWT.getPlatform().equals("win32");


	protected final AnimatableValueElement element;

	protected final String name;

	protected final AnimatableValue<V> avalue;

	protected V value;

	protected final Rectangle stopwatchArea = new Rectangle(Integer.MIN_VALUE, Integer.MIN_VALUE, 13, 14);

	protected final List<Rectangle> valueArea = Util.newList();

	protected int yKeyframe = Integer.MIN_VALUE;

	protected V originalValue;

	protected int valueIndex;

	protected DragGestureEditor dragGestureEditor;

	protected boolean canInlineEdit;

	protected Element[] children;


	public AnimatableValueElementDelegate(
			AnimatableValueElement element, String name, AnimatableValue<V> avalue) {

		this.element = element;
		this.name = name;
		this.avalue = avalue;
	}

	public Image getColumnImage(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return ImageUtil.getStopwatchIcon(avalue.hasKeyframe());

			default:
				return null;
		}
	}

	public String getColumnText(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return name;

			case LayerCompositionView.VALUE_COL:
				// ここでnullや空文字を返すと、なぜかCellEditorがすぐにフォーカスアウトしてしまう場合がある。
				return " ";

			default:
				return null;
		}
	}

	public boolean hasChildren() {
		return hasExpression();
	}

	public Element[] getChildren() {
		if (hasExpression()) {
			if (children == null) {
				children = new Element[] { new ExpressionElement() };
			}
			return children;
		} else {
			return Element.EMPTY_CHILDREN;
		}
	}

	public void paintColumn(Event event) {
		switch (event.index) {
			case LayerCompositionView.NAME_COL:
				stopwatchArea.x = event.x + 5;
				stopwatchArea.y = event.y + (WIN32 ? 1 : 4);
				break;

			case LayerCompositionView.VALUE_COL:
				updateValue();

				valueArea.clear();

				GC gc = event.gc;
				int x = event.x + 5;
				int y = event.y;
				int height = ((TreeItem) event.item).getBounds(event.index).height;
				boolean focused = ((event.detail & SWT.SELECTED) != 0 && ((Control) event.widget).isFocusControl());

				drawValue(gc, x, y, height, focused);
				break;

			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				yKeyframe = tm.drawKeyframes(event, avalue);
				break;
		}
	}

	public void updateCursor(MouseEvent event, int columnIndex) {
		Cursor cursor = null;
		switch (columnIndex) {
			case LayerCompositionView.VALUE_COL:
				for (Rectangle r : valueArea) {
					if (r.contains(event.x, event.y)) {
						cursor = event.display.getSystemCursor(SWT.CURSOR_HAND);
						break;
					}
				}
				break;
		}
		element.viewer.getTree().setCursor(cursor);
	}

	public void mouseDown(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				if (stopwatchArea.contains(event.x, event.y)) {
					toggleStopwatchButon();
				}
				break;

			case LayerCompositionView.VALUE_COL:
				if (event.button == 1) {
					for (int i = 0, n = valueArea.size(); i < n; ++i) {
						Rectangle r = valueArea.get(i);
						if (r.contains(event.x, event.y)) {
							if (!canDragGestureEdit()) {
								valueIndex = i;
								openInlineEditor();
							} else if (dragGestureEditor == null) {
								originalValue = value;
								valueIndex = i;
								dragGestureEditor = new DragGestureEditor(event);
							}
							break;
						}
					}
				}
				break;

			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				tm.mouseDown(event, yKeyframe, element, avalue);
				break;
		}
	}

	public boolean canEdit(int columnIndex) {
		return false;
	}

	public CellEditor getCellEditor(int columnIndex) {
		return null;
	}

	public Object getCellEditorValue(int columnIndex) {
		return null;
	}

	public void setCellEditorValue(int columnIndex, Object value) {
	}

	public boolean updateValue() {
		@SuppressWarnings("unchecked")
		Map<AnimatableValue<?>, ?> animatableValues = (Map<AnimatableValue<?>, ?>)
				element.viewer.getData(LayerCompositionView.ANIMATABLE_VALUES);

		@SuppressWarnings("unchecked")
		V newValue = (V) animatableValues.get(avalue);

		if ((value != null && !value.equals(newValue)) || (value == null && newValue != null)) {
			value = newValue;
			return true;
		} else {
			return false;
		}
	}

	public boolean hasExpression() {
		return (avalue.getExpression() != null);
	}

	protected void openInlineEditor() {
		element.viewer.getTree().setCursor(null);

		canInlineEdit = true;
		element.viewer.editElement(element, LayerCompositionView.VALUE_COL);
		canInlineEdit = false;
	}

	protected void modifyValue(V newValue) {
		modifyValue(newValue, null);
	}

	protected void modifyValue(V newValue, String relation) {
		if (newValue != null && avalue instanceof ArithmeticalAnimatableValue<?>) {
			newValue = ((ArithmeticalAnimatableValue<V>) avalue).clamp(newValue);
		}

		if ((value != null && !value.equals(newValue)) || (value == null && newValue != null)) {
			ProjectManager pm = (ProjectManager) element.viewer.getData(LayerCompositionView.PROJECT_MANAGER);
			TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);

			pm.postOperation(avalue.hasKeyframe()
					? element.createModifyKeyframeOperation(pm, tm.getCurrentTime(), newValue, relation)
					: element.createModifyValueOperation(pm, newValue, relation));
		}
	}

	protected void modifyExpression(String newExpr) {
		if (newExpr != null) {
			newExpr = newExpr.trim();
			if (newExpr.length() == 0) {
				newExpr = null;
			}
		}

		String oldExpr = avalue.getExpression();
		if ((oldExpr != null && !oldExpr.equals(newExpr)) || (oldExpr == null && newExpr != null)) {
			ProjectManager pm = (ProjectManager) element.viewer.getData(LayerCompositionView.PROJECT_MANAGER);
			ProjectOperation operation = element.createModifyExpressionOperation(pm, newExpr);
			pm.postOperation(operation);
		}
	}

	protected void toggleStopwatchButon() {
		ProjectManager pm = (ProjectManager) element.viewer.getData(LayerCompositionView.PROJECT_MANAGER);
		TimelineManager tm = (TimelineManager) element.viewer.getData(LayerCompositionView.TIMELINE_MANAGER);

		pm.postOperation(avalue.hasKeyframe()
				? element.createClearKeyframesOperation(pm)
				: element.createModifyKeyframeOperation(pm, tm.getCurrentTime(), value, null));
	}

	protected abstract void drawValue(GC gc, int x, int y, int height, boolean focused);

	protected abstract boolean canDragGestureEdit();

	protected abstract V dragGesture(double dx, double dy);


	protected void setForeground(GC gc, boolean focused) {
		gc.setForeground(gc.getDevice().getSystemColor(
				focused ? SWT.COLOR_WHITE : SWT.COLOR_BLACK));
	}

	protected void setValueForeground(GC gc, boolean focused) {
		gc.setForeground(gc.getDevice().getSystemColor(
				focused ? SWT.COLOR_WHITE : SWT.COLOR_DARK_BLUE));
	}

	protected int drawValue(
			GC gc, int x, int y, int height, boolean focused, String value) {

		setValueForeground(gc, focused);

		Point extent = gc.textExtent(value, SWT.DRAW_TRANSPARENT);

		y += (height - extent.y) / 2;
		gc.drawString(value, x, y, true);

		valueArea.add(new Rectangle(x, y, extent.x, extent.y));

		y += extent.y;
		gc.setLineDash(UNDERLINE_DASH);
		if (COCOA) {
			gc.drawLine(x, y - 1, x + extent.x, y - 1);
		} else if (WIN32) {
			gc.drawLine(x, y, x + extent.x - 2, y);
		} else {
			gc.drawLine(x, y, x + extent.x, y);
		}

		return x + extent.x;
	}

	protected int drawValue(
			GC gc, int x, int y, int height, boolean focused, Double value, int precision, String unit) {

		x = drawValue(gc, x, y, height, focused, String.format("%." + precision + "f", value));

		if (unit != null) {
			x = drawString(gc, x + 1, y, height, unit);
		}

		return x;
	}

	protected int drawValue(
			GC gc, int x, int y, int height, boolean focused, Integer value, String unit) {

		x = drawValue(gc, x, y, height, focused, value.toString());

		if (unit != null) {
			x = drawString(gc, x + 1, y, height, unit);
		}

		return x;
	}

	protected int drawString(GC gc, int x, int y, int height, String str) {
		Point extent = gc.textExtent(str, SWT.DRAW_TRANSPARENT);

		y += (height - extent.y) / 2;
		gc.drawString(str, x, y, true);

		return x + extent.x;
	}

	protected String formatValue(Double value, int precision) {
		if (precision < 0) throw new IllegalArgumentException();
		String s = String.format("%." + precision + "f", value);
		int dotIndex = s.indexOf('.');
		if (dotIndex != -1) {
			for (int i = s.length()-1; i >= dotIndex; --i) {
				char c = s.charAt(i);
				if (c == '0') continue;
				return s.substring(0, (c == '.') ? i : i+1);
			}
		}
		return s;
	}


	protected class DragGestureEditor {

		private final String relation = Util.randomId();

		private final long downTime;

		private final Point downPoint;

		private boolean dragDetected;

		private Point detectPoint;

		private Point prevPoint;

		private double[] currentPoint;


		protected DragGestureEditor(MouseEvent event) {
			Control control = (Control) event.widget;
			downTime = System.currentTimeMillis();
			downPoint = control.toDisplay(event.x, event.y);

			init(control);
		}

		private void init(final Control control) {
			final Display display = control.getDisplay();

			Listener listener = new Listener() {
				public void handleEvent(Event e) {
					switch (e.type) {
						case SWT.MouseMove:
							if (!control.isDisposed()) {
								Point pt = display.getCursorLocation();

								if (!dragDetected) {
									dragDetected = (System.currentTimeMillis() - downTime > 100)
											&& (Math.abs(pt.x - downPoint.x) > 3 || Math.abs(pt.y - downPoint.y) > 3);
									if (dragDetected) {
										currentPoint = new double[] { pt.x, pt.y };
										prevPoint = detectPoint = pt;
									}
								}

								if (dragDetected) {
									double deltaScale = (e.stateMask & SWT.MOD2) != 0 ? 10 :
														(e.stateMask & SWT.MOD1) != 0 ? 0.1 : 1;
									currentPoint[0] += (pt.x - prevPoint.x) * deltaScale;
									currentPoint[1] += (pt.y - prevPoint.y) * deltaScale;

									prevPoint = pt;

									V newValue = dragGesture(currentPoint[0] - detectPoint.x, currentPoint[1] - detectPoint.y);
									modifyValue(newValue, relation);
								}

								break;
							}
							// fall through

						case SWT.MouseUp:
						case SWT.Deactivate:
							dragGestureEditor = null;

							display.removeFilter(SWT.MouseMove, this);
							display.removeFilter(SWT.MouseUp, this);
							display.removeFilter(SWT.Deactivate, this);

							if (e.type == SWT.MouseUp && !control.isDisposed() && !dragDetected) {
								openInlineEditor();
							}
							break;
					}
				}
			};

			display.addFilter(SWT.MouseMove, listener);
			display.addFilter(SWT.MouseUp, listener);
			display.addFilter(SWT.Deactivate, listener);
		}

	}


	public class ExpressionElement extends Element {

		private static final String EDITOR_KEY = "ExpressionElement.TextCellEditor";


		public ExpressionElement() {
			super(element);
		}

		public String getColumnText(int columnIndex) {
			switch (columnIndex) {
				case LayerCompositionView.NAME_COL:
					return "エクスプレッション：" + name;

				case LayerCompositionView.TIMELINE_COL:
					return avalue.getExpression();

				default:
					return null;
			}
		}

		public void mouseDown(MouseEvent event, int columnIndex) {
			super.mouseDown(event, columnIndex);

			// WIN32では選択中の行でしかエディタが起動しないので、プログラム上からエディタを起動する。
			// (おそらくFULL_SELECTIONにすれば選択中でなくても(行のどこでもクリックすれば行が選択されるので、結果的に)エディタは起動するはずだが)

			if (WIN32 && columnIndex == LayerCompositionView.TIMELINE_COL && event.button == 1) {
				viewer.editElement(this, LayerCompositionView.TIMELINE_COL);
			}
		}

		public boolean canEdit(int columnIndex) {
			if (columnIndex == LayerCompositionView.TIMELINE_COL) {
				return true;
			}
			return false;
		}

		public CellEditor getCellEditor(int columnIndex) {
			if (columnIndex == LayerCompositionView.TIMELINE_COL) {
				TextCellEditor editor = (TextCellEditor) viewer.getData(EDITOR_KEY);
				if (editor == null) {
					editor = new TextCellEditor(viewer.getTree(), SWT.SINGLE | SWT.BORDER);
					Control control = editor.getControl();
					control.setBackground(control.getDisplay().getSystemColor(SWT.COLOR_WHITE));
					viewer.setData(EDITOR_KEY, editor);
				}
				return editor;
			}
			return null;
		}

		public Object getCellEditorValue(int columnIndex) {
			if (columnIndex == LayerCompositionView.TIMELINE_COL) {
				return avalue.getExpression();
			}
			return null;
		}

		public void setCellEditorValue(int columnIndex, Object value) {
			if (columnIndex == LayerCompositionView.TIMELINE_COL) {
				modifyExpression((String) value);
			}
		}

	}

}
