/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.Time;

public abstract class AbstractAnimatableValue<V> implements AnimatableValue<V> {

	protected V _defaultValue;

	protected V _staticValue;

	protected final SortedMap<Time, Keyframe<V>> _keyframes;

	protected String _expression;


	public AbstractAnimatableValue(V staticValue, Collection<Keyframe<V>> keyframes, String expression) {
		super();
		_defaultValue = staticValue;
		_staticValue = staticValue;
		_keyframes = Util.newSortedMap();
		_expression = expression;

		if (keyframes != null) {
			for (Keyframe<V> k : keyframes) {
				putKeyframe(k);
			}
		}
	}

	public AbstractAnimatableValue(V defaultValue) {
		this(defaultValue, null, null);
	}

	public void copyConfigurationFrom(AbstractAnimatableValue<V> src) {
		if (this.getClass() != src.getClass()) {
			throw new IllegalArgumentException("class does not match: " + src.getClass());
		}
		_defaultValue = src._defaultValue;
	}

	public V value(RenderContext renderContext) {
		if (_expression != null && !renderContext.checkCyclicEvaluation(this)) {
			try {
				V value = renderContext.evaluateExpression(this);
				if (value != null) {
					return value;
				}
				throw new NullPointerException();
			} catch (Exception e) {
				// TODO エラーが発生しているかどうかを示すフラグを立てる。_expressionを変更した後はそのフラグはリセットする。
				// TODO そのフラグを立てたとき、つまり初回だけエクスプレッションエラーをGUIに通知する仕組みを作る
				e.printStackTrace();
			}
		}
		return interpolate(renderContext.getTime());
	}

	public V valueAtTime(final Time time, final RenderContext renderContext) {
		if (_expression != null && !renderContext.checkCyclicEvaluation(this)) {
			try {
				V value = renderContext.saveAndExecute(new WrappedOperation<V>() {
					public V execute() {
						renderContext.setTime(time);
						return renderContext.evaluateExpression(AbstractAnimatableValue.this);
					}
				});
				if (value != null) {
					return value;
				}
				throw new NullPointerException();
			} catch (Exception e) {
				// TODO エラーが発生しているかどうかを示すフラグを立てる。_expressionを変更した後はそのフラグはリセットする。
				// TODO そのフラグを立てたとき、つまり初回だけエクスプレッションエラーをGUIに通知する仕組みを作る
				e.printStackTrace();
			}
		}
		return interpolate(time);
	}

	public void values(
			IObjectArray<V> values, Time sampleDuration, int evalResolution,
			RenderContext renderContext) {

		valuesAtTime(values, sampleDuration, evalResolution,
				renderContext.getTime(), renderContext);
	}

	public void valuesAtTime(
			final IObjectArray<V> values, final Time sampleDuration, final int evalResolution,
			final Time time, final RenderContext renderContext) {

		final Object[] array = values.getArray();
		final int len = values.getLength();

		if (_expression != null && !renderContext.checkCyclicEvaluation(this)) {
			try {
				renderContext.saveAndExecute(new WrappedOperation<Object>() {
					public Object execute() {
						V v = null;
						for (int i = 0; i < len; ++i) {
							if ((i % evalResolution) == 0) {
								Time t = time.add(new Time(sampleDuration.timeValue*i, sampleDuration.timeScale));
								renderContext.setTime(t);
								v = renderContext.evaluateExpression(AbstractAnimatableValue.this);
							}
							array[i] = v;
						}
						return null;
					}
				});
				return;
			} catch (Exception e) {
				// TODO エラーが発生しているかどうかを示すフラグを立てる。_expressionを変更した後はそのフラグはリセットする。
				// TODO そのフラグを立てたとき、つまり初回だけエクスプレッションエラーをGUIに通知する仕組みを作る
				e.printStackTrace();
			}
		}

		V v = null;
		for (int i = 0; i < len; ++i) {
			if ((i % evalResolution) == 0) {
				Time t = time.add(new Time(sampleDuration.timeValue*i, sampleDuration.timeScale));
				v = interpolate(t);
			}
			array[i] = v;
		}
	}

	protected V interpolate(Time time) {
		if (_keyframes.isEmpty()) {
			return _staticValue;
		}

		Keyframe<V> k1 = null;
		Keyframe<V> k2 = null;

		Iterator<Map.Entry<Time, Keyframe<V>>> tail = _keyframes.tailMap(time).entrySet().iterator();
		if (tail.hasNext()) {
			Map.Entry<Time, Keyframe<V>> entry = tail.next();
			if (entry.getKey().equals(time)) {
				k1 = entry.getValue();
				if (tail.hasNext()) {
					k2 = tail.next().getValue();
				}
			} else {
				k2 = entry.getValue();
			}
		}
		if (k1 == null) {
			SortedMap<Time, Keyframe<V>> headMap = _keyframes.headMap(time);
			if (!headMap.isEmpty()) {
				k1 = headMap.get(headMap.lastKey());
			}
		}

		if (k1 == null) {
			return k2.value;
		} if (k2 == null) {
			return k1.value;
		}

		return interpolate(time, k1, k2, tail);
	}

	protected V interpolate(
			Time time, Keyframe<V> k1, Keyframe<V> k2, Iterator<Map.Entry<Time, Keyframe<V>>> tail) {

		switch (k1.interpolation) {
			case HOLD:
				return k1.value;

			default:
				throw new UnsupportedOperationException(k1.interpolation.name());
		}
	}

	public V getStaticValue() {
		return _staticValue;
	}

	public boolean hasKeyframe() {
		return !_keyframes.isEmpty();
	}

	public Keyframe<V> getKeyframe(Time time) {
		return _keyframes.get(time);
	}

	public Collection<Keyframe<V>> getKeyframes() {
		return Collections.unmodifiableCollection(_keyframes.values());
	}

	public Collection<Keyframe<V>> getKeyframes(Time fromTime, Time toTime) {
		return Collections.unmodifiableCollection(_keyframes.subMap(fromTime, toTime).values());
	}

	public SortedMap<Time, Keyframe<V>> headKeyframeMap(Time toTime) {
		return Collections.unmodifiableSortedMap(_keyframes.headMap(toTime));
	}

	public SortedMap<Time, Keyframe<V>> tailKeyframeMap(Time fromTime) {
		return Collections.unmodifiableSortedMap(_keyframes.tailMap(fromTime));
	}

	public Keyframe<V> putKeyframe(Keyframe<V> keyframe) {
		if (!getAvailableInterpolations().contains(keyframe.interpolation)) {
			throw new IllegalArgumentException("unsupported interpolation: " + keyframe.interpolation.name());
		}
		return _keyframes.put(keyframe.time, keyframe);
	}

	public Keyframe<V> putKeyframe(Time time, V value, Interpolation interpolation) {
		return putKeyframe(new Keyframe<V>(time, value, interpolation));
	}

	public Keyframe<V> removeKeyframe(Time time) {
		Keyframe<V> removed = _keyframes.remove(time);
		if (_keyframes.isEmpty() && removed != null) {
			_staticValue = removed.value;
		}
		return removed;
	}

	public void clearKeyframes() {
		if (!_keyframes.isEmpty()) {
			clearKeyframes(_keyframes.get(_keyframes.firstKey()).value);
		}
	}

	public void clearKeyframes(V staticValue) {
		_staticValue = staticValue;
		_keyframes.clear();
	}

	public void reset() {
		reset(_defaultValue);
	}

	public void reset(V staticValue) {
		clearKeyframes(staticValue);
		_expression = null;
	}

	public String getExpression() {
		return _expression;
	}

	public void setExpression(String expression) {
		_expression = expression;
	}

	private static final Set<Interpolation> INTERPOLATIONS = Util.unmodifiableSet(Interpolation.HOLD);

	@JSONHint(ignore=true)
	public Set<Interpolation> getAvailableInterpolations() {
		return INTERPOLATIONS;
	}

	@JSONHint(ignore=true)
	public Interpolation getDefaultInterpolation() {
		return Interpolation.HOLD;
	}

	protected V valueAtTime(double t, RenderContext renderContext) {
		int timeScale = renderContext.getTime().timeScale;
		Time time = new Time((long) (t * timeScale), timeScale); 
		return valueAtTime(time, renderContext);
	}

}
