/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Collections;
import java.util.List;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.core.AbstractTransformableLayer;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.AnimatableVec2d;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.BlendMode;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.CollapseTransformation;
import ch.kuramo.javie.core.DepthBuffer;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.RenderContext;
import ch.kuramo.javie.core.TrackMatte;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VectorMediaInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.VideoLayerBuffer;
import ch.kuramo.javie.core.VideoLayerComposer;
import ch.kuramo.javie.core.VideoLayerRenderer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.AudioEffectPipeline;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.VideoEffectPipeline;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public abstract class AbstractMediaLayer extends AbstractTransformableLayer implements MediaLayer {

	private boolean _videoNature;

	private boolean _audioNature;

	private boolean _videoEnabled;

	private boolean _audioEnabled;

	private boolean _effectsEnabled = true;

	private List<Effect> _effects = Util.newList();

	private TrackMatte _trackMatte = TrackMatte.NONE;

	private AnimatableVec2d _depthBase = new AnimatableVec2d(new Vec2d(0, 0));

	private AnimatableString _intersectionGroup = new AnimatableString("");

	private AnimatableDouble _opacity = new AnimatableDouble(100d, 0d, 100d);

	private BlendMode _blendMode = BlendMode.NORMAL;

	private AnimatableVec2d _audioLevels = new AnimatableVec2d(Vec2d.ZERO, new Vec2d(-48, -48), new Vec2d(48, 48));

	private AnimatableDouble _timeRemap = new AnimatableDouble(0.0);

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private VideoEffectPipeline _vePipeline;

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private AudioEffectPipeline _aePipeline;

	@Inject
	private IArrayPools _arrayPools;


	@Override
	public void initialize() {
		throw new UnsupportedOperationException("Use initialize(boolean, boolean) method instead.");
	}

	protected void initialize(boolean videoAvailable, boolean audioAvailable) {
		if (!videoAvailable && !audioAvailable) {
			throw new IllegalArgumentException();
		}

		super.initialize();

		if (videoAvailable) {
			_videoNature = true;
			_videoEnabled = true;
		}

		if (audioAvailable) {
			_audioNature = true;
			_audioEnabled = true;
		}
	}

	public boolean isVideoNature() {
		return _videoNature;
	}

	public void setVideoNature(boolean videoNature) {
		_videoNature = videoNature;
	}

	public boolean isAudioNature() {
		return _audioNature;
	}

	public void setAudioNature(boolean audioNature) {
		_audioNature = audioNature;
	}

	public boolean isVideoEnabled() {
		return _videoEnabled;
	}

	public void setVideoEnabled(boolean enabled) {
		_videoEnabled = enabled;
	}

	public boolean isAudioEnabled() {
		return _audioEnabled;
	}

	public void setAudioEnabled(boolean enabled) {
		_audioEnabled = enabled;
	}

	public boolean isEffectsEnabled() {
		return _effectsEnabled;
	}

	public void setEffectsEnabled(boolean effectsEnabled) {
		_effectsEnabled = effectsEnabled;
	}

	public List<Effect> getEffects() {
		return _effects;
	}

	public void setEffects(List<Effect> effects) {
		_effects = effects;
	}

	public TrackMatte getTrackMatte() {
		return _trackMatte;
	}

	public void setTrackMatte(TrackMatte trackMatte) {
		_trackMatte = trackMatte;
	}

	public AnimatableVec2d getDepthBase() {
		return _depthBase;
	}

	public void setDepthBase(AnimatableVec2d depthBase) {
		depthBase.copyConfigurationFrom(_depthBase);
		_depthBase = depthBase;
	}

	public AnimatableString getIntersectionGroup() {
		return _intersectionGroup;
	}

	public void setIntersectionGroup(AnimatableString intersectionGroup) {
		intersectionGroup.copyConfigurationFrom(_intersectionGroup);
		_intersectionGroup = intersectionGroup;
	}

	public AnimatableDouble getOpacity() {
		return _opacity;
	}

	public void setOpacity(AnimatableDouble opacity) {
		opacity.copyConfigurationFrom(_opacity);
		_opacity = opacity;
	}

	public BlendMode getBlendMode() {
		return _blendMode;
	}

	public void setBlendMode(BlendMode blendMode) {
		_blendMode = blendMode;
	}

	public AnimatableVec2d getAudioLevels() {
		return _audioLevels;
	}

	public void setAudioLevels(AnimatableVec2d audioLevels) {
		audioLevels.copyConfigurationFrom(_audioLevels);
		_audioLevels = audioLevels;
	}

	public AnimatableDouble getTimeRemap() {
		return _timeRemap;
	}

	public void setTimeRemap(AnimatableDouble timeRemap) {
		timeRemap.copyConfigurationFrom(_timeRemap);
		_timeRemap = timeRemap;
	}

	public boolean isVectorLayer() {
		return (getMediaInput() instanceof VectorMediaInput);
	}

	@Override
	public void prepareExpression(ExpressionScope scope) {
		super.prepareExpression(scope);

		scope.assignTo(_depthBase);
		scope.assignTo(_intersectionGroup);
		scope.assignTo(_opacity);
		scope.assignTo(_audioLevels);
		scope.assignTo(_timeRemap);

		for (Effect effect : _effects) {
			effect.prepareExpression(scope.clone());
		}
	}

	public void setupVideoRenderer(
			VideoLayerComposer composer, Camera camera, CollapseTransformation ct) {

		// トラックマット適用時にこのチェックがあるとまずいので。同じチェックはコンポ側でもしている。
		//if (!LayerNature.isVideoEnabled(this)) {
		//	return;
		//}

		MediaInput input = getMediaInput();
		if (input == null || !input.isVideoAvailable()) {
			return;
		}

		if (isVectorLayer() && LayerNature.isCTCR(AbstractMediaLayer.this)) {
			setupVideoRendererCR(composer, camera, ct);
		} else {
			setupVideoRendererNormal(composer, camera, ct);
		}
	}

	private Time calcVideoMediaTime(MediaInput input) {
		Time time;

		if (LayerNature.isTimeRemapEnabled(this)) {
			double t0 = _timeRemap.value(_vrContext);

			Time ctxTime = _vrContext.getTime();
			double t1 = _timeRemap.valueAtTime(new Time(ctxTime.timeValue + 1, ctxTime.timeScale), _vrContext);

			Time frameDuration = input.getVideoFrameDuration();
			int timeScale = frameDuration.timeScale;

			time = new Time(Math.round(t0 * timeScale), timeScale);

			// 変化量が負の時のときは１フレーム分ずらす。
			if (t0 > t1) {
				time = time.subtract(frameDuration);
			}

		} else {
			time = _vrContext.getTime().subtract(getStartTime());

			double rate = getRate();
			if (rate != 1.0) {
				time = new Time(Math.round(time.timeValue * rate), time.timeScale);

				if (rate < 0.0) {
					Time duration = input.getDuration();
					if (duration != null) {
						time = time.add(duration);

						// 逆再生時は１フレーム分ずらす。
						Time frameDuration = input.getVideoFrameDuration();
						time = time.subtract(frameDuration);

					} else {
						duration = getOutPoint().subtract(getStartTime());
						time = time.add(duration);
					}
				}
			}
		}

		return time;
	}

	private void setupVideoRendererNormal(
			VideoLayerComposer composer, final Camera camera, final CollapseTransformation ct) {

		final MediaInput input = getMediaInput();
		final RenderResolution resolution = _vrContext.getRenderResolution();

		WrappedOperation<VideoBounds> inputBoundsOperation = new WrappedOperation<VideoBounds>() {
			public VideoBounds execute() {
				return resolution.scale(input.getVideoFrameBounds());
			}
		};

		VideoBounds bounds = _vePipeline.getVideoBounds(
				_effectsEnabled ? _effects : Collections.<Effect>emptyList(), inputBoundsOperation);
		Vec2d offset = new Vec2d(-bounds.x/resolution.scale, -bounds.y/resolution.scale);

		final double[] mvMatrix = new double[16];
		final double[] prjMatrix = new double[16];
		createMatrix(camera, ct, offset, mvMatrix, prjMatrix);

		final double opacity;
		if (ct == null) {
			opacity = _opacity.value(_vrContext) / 100d;
		} else {
			opacity = _opacity.value(_vrContext) / 100d * ct.getOpacity();
		}

		VideoLayerRenderer r = new VideoLayerRenderer() {
			public VideoLayerBuffer render(boolean withDepthBuffer) {
				VideoBuffer effectedBuffer = null;
				try {
					WrappedOperation<VideoBuffer> inputBufferOperation = new WrappedOperation<VideoBuffer>() {
						public VideoBuffer execute() {
							return input.getVideoFrameImage(calcVideoMediaTime(input));
						}
					};

					effectedBuffer = _vePipeline.doVideoEffects(
							_effectsEnabled ? _effects : Collections.<Effect>emptyList(), inputBufferOperation);

					VideoBuffer trnsfrmdBuffer = _vrSupport.createVideoBuffer(_vrContext.getColorMode(), camera.getViewportSize());
					trnsfrmdBuffer.allocateAsTexture();
					trnsfrmdBuffer.clear();

					DepthBuffer depthBuffer = withDepthBuffer ? _vrSupport.createDepthBuffer(camera.getViewportSize()) : null;

					_vrSupport.transform(effectedBuffer, trnsfrmdBuffer, depthBuffer, mvMatrix, prjMatrix);

					// TODO 親コンポがコラップスしている場合、プリコンポレイヤーのエフェクトはここ？

					return new VideoLayerBufferImpl(trnsfrmdBuffer, depthBuffer, _blendMode, opacity);

				} finally {
					if (effectedBuffer != null) {
						effectedBuffer.dispose();
					}
				}
			}
		};

		if (LayerNature.isThreeD(this)) {
			Vec2d depthBase2d = _depthBase.value(_vrContext);
			Vec3d depthBase = new Vec3d(depthBase2d.x + offset.x, depthBase2d.y + offset.y);
			depthBase = resolution.scale(depthBase);

			Size2i vp = camera.getViewportSize();
			Vec3d[] vertices = new Vec3d[] {
				_vrSupport.project(depthBase, mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y + bounds.height), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y + bounds.height), mvMatrix, prjMatrix, vp)
			};
			composer.add3D(r, vertices, _intersectionGroup.value(_vrContext));

		} else {
			composer.add2D(r);
		}
	}

	private void setupVideoRendererCR(
			VideoLayerComposer composer, final Camera camera, final CollapseTransformation ct) {

		final VectorMediaInput input = (VectorMediaInput) getMediaInput();
		RenderResolution resolution = _vrContext.getRenderResolution();

		VideoBounds bounds = resolution.scale(input.getVideoFrameBounds());

		final double[] mvMatrix = new double[16];
		final double[] prjMatrix = new double[16];
		createMatrix(camera, ct, null, mvMatrix, prjMatrix);

		final double opacity;
		if (ct == null) {
			opacity = _opacity.value(_vrContext) / 100d;
		} else {
			opacity = _opacity.value(_vrContext) / 100d * ct.getOpacity();
		}

		VideoLayerRenderer r = new VideoLayerRenderer() {
			public VideoLayerBuffer render(boolean withDepthBuffer) {
				final VideoBuffer rasterBuffer = _vrSupport.createVideoBuffer(_vrContext.getColorMode(), camera.getViewportSize());
				rasterBuffer.allocateAsTexture();
				rasterBuffer.clear();

				// TODO 連続ラスタライズの場合、デプスバッファはここで作成せずに全てのエフェクトが終わった後で計算する。
				final DepthBuffer depthBuffer = withDepthBuffer ? _vrSupport.createDepthBuffer(camera.getViewportSize()) : null;

				_vrContext.saveAndExecute(new WrappedOperation<Object>() {
					public Object execute() {
						_vrContext.setTime(calcVideoMediaTime(input));
						input.rasterize(rasterBuffer, depthBuffer, mvMatrix, prjMatrix);
						return null;
					}
				});

				// TODO このレイヤーのエフェクト
				// TODO 親コンポがコラップスしている場合、プリコンポレイヤーのエフェクトはここ？

				return new VideoLayerBufferImpl(rasterBuffer, depthBuffer, _blendMode, opacity);
			}
		};

		if (LayerNature.isThreeD(this)) {
			Vec2d depthBase2d = _depthBase.value(_vrContext);
			Vec3d depthBase = new Vec3d(depthBase2d.x, depthBase2d.y);
			depthBase = resolution.scale(depthBase);

			Size2i vp = camera.getViewportSize();
			Vec3d[] vertices = new Vec3d[] {
				_vrSupport.project(depthBase, mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y + bounds.height), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y + bounds.height), mvMatrix, prjMatrix, vp)
			};
			composer.add3D(r, vertices, _intersectionGroup.value(_vrContext));

		} else {
			composer.add2D(r);
		}
	}

	private void createMatrix(
			final Camera camera, final CollapseTransformation ct, final Vec2d offset,
			final double[] mvMatrix, final double[] prjMatrix) {

		WrappedOperation<Object> operation = new WrappedOperation<Object>() {
			public Object execute() {
				if (camera == null) {
					_vrSupport.resetMatrix();
				} else if (LayerNature.isThreeD(AbstractMediaLayer.this)) {
					_vrSupport.setMatrix(camera.getModelView3D(), camera.getProjection3D());
				} else {
					_vrSupport.setMatrix(camera.getModelView2D(), camera.getProjection2D());
				}

				LayerMatrixUtil.multModelViewMatrix(AbstractMediaLayer.this, offset, ct, _vrContext, _vrSupport);

				_vrSupport.getMatrix(mvMatrix, prjMatrix);
				return null;
			}
		};

		_vrSupport.pushMatrixAndExecute(operation);
	}

	private AudioBuffer getTimeRemappedAudioChunk(MediaInput input) {
		AudioBuffer ab1 = null;
		AudioBuffer ab2 = null;
		IObjectArray<Double> times = null;

		try {
			AudioMode audioMode = _arContext.getAudioMode();
			int sampleRate = audioMode.sampleRate;

			ab1 = _arSupport.createAudioBuffer();

			times = _arrayPools.getObjectArray(ab1.getFrameCount());
			_timeRemap.values(times, audioMode.sampleDuration, _arContext.getEvaluationResolution(), _arContext);


			// 音声データを取得する範囲を求める。
			double min = Double.POSITIVE_INFINITY, max = Double.NEGATIVE_INFINITY;
			Object[] timeArray = times.getArray();
			for (int i = 0, n = times.getLength(); i < n; ++i) {
				min = Math.min(min, (Double) timeArray[i]);
				max = Math.max(max, (Double) timeArray[i]);
			}
			long timeValueMin = (long) Math.floor(min*sampleRate);
			long timeValueMax = (long) Math.ceil(max*sampleRate);


			// 上で求めた範囲の音声データをMediaInputから取得。
			_arContext.setFrameCount((int)(timeValueMax - timeValueMin) + 1);

			ab2 = input.getAudioChunk(new Time(timeValueMin, sampleRate));
			if (ab2 == null) return null;


			// タイムリマップの計算値に従ってリサンプルする。
			Object data1 = ab1.getData();
			Object data2 = ab2.getData();
			int n1 = ab1.getFrameCount();

			switch (audioMode.dataType) {
				case SHORT: {
					short[] array1 = (short[]) data1, array2 = (short[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (int)(Math.round((Double)timeArray[i]*sampleRate) - timeValueMin);
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case INT: {
					int[] array1 = (int[]) data1, array2 = (int[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (int)(Math.round((Double)timeArray[i]*sampleRate) - timeValueMin);
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case FLOAT: {
					float[] array1 = (float[]) data1, array2 = (float[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (int)(Math.round((Double)timeArray[i]*sampleRate) - timeValueMin);
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}


			AudioBuffer result = ab1;
			ab1 = null;
			return result;

		} finally {
			if (ab1 != null) ab1.dispose();
			if (ab2 != null) ab2.dispose();
			if (times != null) times.release();
		}
	}

	private AudioBuffer getRateChangedAudioChunk(MediaInput input, Time time, double rate) {
		AudioBuffer ab1 = null;
		AudioBuffer ab2 = null;

		try {
			AudioMode audioMode = _arContext.getAudioMode();
			int sampleRate = audioMode.sampleRate;

			ab1 = _arSupport.createAudioBuffer();

			long timeValueMin;
			long timeValueMax;
			double duration = input.getDuration().toSecond();

			if (rate >= 0) {
				timeValueMin = (long) Math.floor(time.toSecond() * sampleRate * rate);

				Time t = time.add(new Time(_arContext.getFrameCount(), sampleRate));
				timeValueMax = (long) Math.ceil(t.toSecond() * sampleRate * rate);

			} else {
				timeValueMax =  (long) Math.ceil(duration * sampleRate + time.toSecond() * sampleRate * rate);

				Time t = time.add(new Time(_arContext.getFrameCount(), sampleRate));
				timeValueMin = (long) Math.floor(duration * sampleRate + t.toSecond() * sampleRate * rate);
			}

			_arContext.setFrameCount((int)(timeValueMax - timeValueMin) + 1);

			ab2 = input.getAudioChunk(new Time(timeValueMin, sampleRate));
			if (ab2 == null) return null;


			Object data1 = ab1.getData();
			Object data2 = ab2.getData();
			int n1 = ab1.getFrameCount();

			switch (audioMode.dataType) {
				case SHORT: {
					short[] array1 = (short[]) data1, array2 = (short[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (rate >= 0) ? (int)(Math.round((time.toSecond() * sampleRate + i) * rate) - timeValueMin)
											: (int)(Math.round(duration * sampleRate + (time.toSecond() * sampleRate + i) * rate) - timeValueMin); 
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case INT: {
					int[] array1 = (int[]) data1, array2 = (int[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (rate >= 0) ? (int)(Math.round((time.toSecond() * sampleRate + i) * rate) - timeValueMin)
											: (int)(Math.round(duration * sampleRate + (time.toSecond() * sampleRate + i) * rate) - timeValueMin); 
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case FLOAT: {
					float[] array1 = (float[]) data1, array2 = (float[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (rate >= 0) ? (int)(Math.round((time.toSecond() * sampleRate + i) * rate) - timeValueMin)
											: (int)(Math.round(duration * sampleRate + (time.toSecond() * sampleRate + i) * rate) - timeValueMin); 
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}


			AudioBuffer result = ab1;
			ab1 = null;
			return result;

		} finally {
			if (ab1 != null) ab1.dispose();
			if (ab2 != null) ab2.dispose();
		}
	}

	public AudioBuffer renderAudioChunk() {
		if (!LayerNature.isAudioEnabled(this)) {
			return null;
		}

		final MediaInput input = getMediaInput();
		if (input == null || !input.isAudioAvailable()) {
			return null;
		}

		WrappedOperation<AudioBuffer> inputOperation = new WrappedOperation<AudioBuffer>() {
			public AudioBuffer execute() {
				if (LayerNature.isTimeRemapEnabled(AbstractMediaLayer.this)) {
					return getTimeRemappedAudioChunk(input);

				} else {
					Time time = _arContext.getTime().subtract(getStartTime());
					double rate = getRate();
					if (rate == 1.0) {
						return input.getAudioChunk(time);
					} else {
						return getRateChangedAudioChunk(input, time, rate);
					}
				}
			}
		};

		AudioBuffer ab = _aePipeline.doAudioEffects(
				_effectsEnabled ? _effects : Collections.<Effect>emptyList(), inputOperation);

		if (_audioLevels.hasKeyframe() || _audioLevels.getExpression() != null
				|| !_audioLevels.getStaticValue().equals(Vec2d.ZERO)) {

			IObjectArray<Vec2d> audioLevels = _arrayPools.getObjectArray(ab.getFrameCount());

			_audioLevels.values(audioLevels, ab.getAudioMode().sampleDuration,
								_arContext.getEvaluationResolution(), _arContext);
			_arSupport.amplify(ab, audioLevels);

			audioLevels.release();
		}

		// インポイントよりも手前とアウトポイント以降はゼロクリアする
		Time time = _arContext.getTime();
		Time sampleDuration = _arContext.getAudioMode().sampleDuration;

		int clearFrameCount = (int) getInPoint().subtract(time).toFrameNumber(sampleDuration);
		if (clearFrameCount > 0) {
			_arSupport.clear(ab, 0, clearFrameCount);
		}

		int clearFrameOffset = (int) getOutPoint().subtract(time).toFrameNumber(sampleDuration);
		clearFrameCount = _arContext.getFrameCount() - clearFrameOffset;
		if (clearFrameCount > 0) {
			_arSupport.clear(ab, clearFrameOffset, clearFrameCount);
		}

		return ab;
	}

	public void afterDecode(Project p, LayerComposition c) throws ProjectDecodeException {
		super.afterDecode(p, c);

		for (Effect effect : _effects) {
			effect.afterDecode(p);
		}
	}

	public Object createExpressionElement(RenderContext renderContext) {
		return new MediaLayerExpressionElement(renderContext);
	}

	public class MediaLayerExpressionElement extends TransformableLayerExpressionElement {

		public MediaLayerExpressionElement(RenderContext renderContext) {
			super(renderContext);
		}

		public Object getDepthBase()			{ return elem(_depthBase); }
		public Object getIntersectionGroup()	{ return elem(_intersectionGroup); }
		public Object getOpacity()				{ return elem(_opacity); }
		public Object getAudioLevels()			{ return elem(_audioLevels); }
		public Object getTimeRemap()			{ return LayerNature.isTimeRemapEnabled(AbstractMediaLayer.this) ? elem(_timeRemap) : null; }

		public Object effect(int index) {
			Effect effect = _effects.get(index - 1);
			return renderContext.getExpressionElement(effect);
		}
	}

}
