/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.UnsupportedAudioFileException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.MediaFileInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;

import com.google.inject.Inject;

public class JavaSoundInput implements MediaFileInput {

	private static final Logger _logger = LoggerFactory.getLogger(JavaSoundInput.class);


	private File _file;

	private Time _duration;

	private AudioMode _audioMode;

	private AudioInputStream _stream;

	private long _streamPosition;

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private IArrayPools _arrayPools;


	public JavaSoundInput() {
		super();
	}

	public boolean initialize(File file) {
		_file = file;

		try {
			AudioInputStream stream = AudioSystem.getAudioInputStream(file);
			try {
				long numFrames = stream.getFrameLength();
				AudioFormat format = stream.getFormat();

				// getFrameLengthはバイト数ではなくフレーム数を返すはずなのに、Macではバイト数を返すバグがあるようである。
				// getFrameLengthが返した値が正しければ、それにgetFrameSizeの値を掛けてもファイルサイズを超えるはずはないが、
				// このバグが発生する場合はファイルサイズを超えることになる。以下ではそのチェックと対処をしている。
				if ((format.getEncoding().equals(AudioFormat.Encoding.PCM_SIGNED)
						|| format.getEncoding().equals(AudioFormat.Encoding.PCM_UNSIGNED))
						&& numFrames * format.getFrameSize() > file.length()) {

					numFrames /= format.getFrameSize();
				}

				float frameRate = format.getFrameRate();
				int frameRateInt = (int) frameRate;
				if (frameRate - frameRateInt != 0) {
					_logger.warn("frame rate is not integer: " + frameRate);
				}
				_duration = new Time(numFrames, frameRateInt);
			} finally {
				stream.close();
			}
		} catch (UnsupportedAudioFileException e) {
			return false;
		} catch (IOException e) {
			return false;
		}

		return true;
	}

	public void dispose() {
		closeStream();
	}

	private void closeStream() {
		if (_stream != null) {
			try {
				_stream.close();
			} catch (IOException e) {
				// ログだけ書き出して無視
				_logger.warn("failed to close AudioInputStream", e);
			}
			_stream = null;
		}
		_streamPosition = 0;
	}

	public synchronized AudioBuffer getAudioChunk(Time mediaTime) {
		try {
			AudioMode audioMode = _arContext.getAudioMode();
			long frameNumber = mediaTime.toFrameNumber(audioMode.sampleDuration);

			if (-frameNumber >= _arContext.getFrameCount() || !mediaTime.before(_duration)) {
				AudioBuffer ab = _arSupport.createAudioBuffer();
				_arSupport.clear(ab);
				return ab;
			}

			if (audioMode != _audioMode || audioMode.frameSize*frameNumber < _streamPosition) {
				closeStream();
			}

			if (_stream == null) {
				_stream = createAudioInputStream();
			}

			return readAudioData(frameNumber);

		} catch (IOException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		} catch (UnsupportedAudioFileException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		}
	}

	private AudioInputStream createAudioInputStream() throws IOException, UnsupportedAudioFileException {
		// サンプルレートと他のパラメータを同時に変換しようとすると失敗するようなので、2段階に分けて変換する。
		//
		// 変換の必要が無い場合、AudioSystem.getAudioInputStream は元のストリームをそのまま返すようなので
		// 以下のように無条件に変換を試みても無駄な変換は発生しないようである。

		_audioMode = _arContext.getAudioMode();
		int bits = _audioMode.sampleSize * 8;
		int ch = _audioMode.channels;
		boolean bigEndian = ByteOrder.nativeOrder().equals(ByteOrder.BIG_ENDIAN);

		AudioInputStream s1 = AudioSystem.getAudioInputStream(_file);

		// サンプルレート以外を変換
		AudioFormat f2 = new AudioFormat(s1.getFormat().getSampleRate(), bits, ch, true, bigEndian);
		AudioInputStream s2 = AudioSystem.getAudioInputStream(f2, s1);

		// サンプルレートを変換
		AudioFormat f3 = new AudioFormat(_audioMode.sampleRate, bits, ch, true, bigEndian);
		return AudioSystem.getAudioInputStream(f3, s2);
	}

	private AudioBuffer readAudioData(long frameNumber) throws IOException {
		AudioBuffer ab = _arSupport.createAudioBuffer();
		AudioMode audioMode = ab.getAudioMode();
		Object data = ab.getData();
		int dataOffset = 0;
		int dataLength = ab.getDataLength();
		int lengthInBytes = ab.getDataLengthInBytes();

		if (frameNumber >= 0) {
			long skip = audioMode.frameSize*frameNumber - _streamPosition;
			if (_stream.skip(skip) != skip) {
				_logger.warn(String.format("skip failed: frameNumber=%d, skip=%d", frameNumber, skip));
				closeStream();
				_arSupport.clear(ab);
				return ab;
			}
			_streamPosition += skip;
		} else if (_streamPosition == 0) {
			_arSupport.clear(ab, 0, (int)-frameNumber);
			dataOffset = (int)-frameNumber * audioMode.channels;
			dataLength -= dataOffset;
			lengthInBytes -= dataOffset * audioMode.sampleSize;

		} else {
			throw new IllegalStateException();
		}

		IArray<byte[]> array = _arrayPools.getByteArray(lengthInBytes);
		try {
			byte[] buffer = array.getArray();

			int readBytes = 0;
			do {
				int n = _stream.read(buffer, readBytes, lengthInBytes - readBytes);
				if (n == -1) {
					Arrays.fill(buffer, readBytes, lengthInBytes, (byte)0);
					break;
				}
				readBytes += n;
				_streamPosition += n;
			} while (readBytes < lengthInBytes);

			switch (audioMode.dataType) {
				case SHORT:
					ByteBuffer.wrap(buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
							.asShortBuffer().get((short[]) data, dataOffset, dataLength);
					break;
				case INT:
					ByteBuffer.wrap(buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
							.asIntBuffer().get((int[]) data, dataOffset, dataLength);
					break;
				case FLOAT:
					for (int i = dataOffset; i < dataLength; ++i) {
						int value =  (buffer[i*4  ] & 0xff)
								  | ((buffer[i*4+1] & 0xff) <<  8)
								  | ((buffer[i*4+2] & 0xff) << 16)
								  | ((buffer[i*4+3] & 0xff) << 24);
						((float[]) data)[i] = (float) value / Integer.MAX_VALUE;
					}
					break;
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}
		} finally {
			array.release();
		}

		return ab;
	}

	public boolean isVideoAvailable() {
		return false;
	}

	public boolean isAudioAvailable() {
		return true;
	}

	public Time getDuration() {
		return _duration;
	}

	public Time getVideoFrameDuration() {
		throw new UnsupportedOperationException("video is not available");
	}

	public VideoBounds getVideoFrameBounds() {
		throw new UnsupportedOperationException("video is not available");
	}

	public VideoBuffer getVideoFrameImage(Time mediaTime) {
		throw new UnsupportedOperationException("video is not available");
	}

}
