/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.List;
import java.util.ListIterator;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.AbstractComposition;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.CollapseTransformation;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.RenderContext;
import ch.kuramo.javie.core.TrackMatte;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.VideoLayerBuffer;
import ch.kuramo.javie.core.VideoLayerComposer;
import ch.kuramo.javie.core.VideoLayerRenderer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

@ProjectElement("layerComposition")
public class LayerCompositionImpl extends AbstractComposition implements LayerComposition {

	private List<Layer> _layers;

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private AudioRenderSupport _arSupport;


	public LayerCompositionImpl() {
		super();
	}

	public void initialize(ColorMode colorMode, Size2i size, Time frameDuration, Time duration) {
		super.initialize(colorMode, size, frameDuration, duration);
		_layers = Util.newList();
	}

	public List<Layer> getLayers() {
		return _layers;
	}

	public void setLayers(List<Layer> layers) {
		_layers = layers;
	}

	public Layer getLayer(String id) {
		for (Layer layer : _layers) {
			if (layer.getId().equals(id)) {
				return layer;
			}
		}
		return null;
	}

	public void afterDecode(Project p) throws ProjectDecodeException {
		for (Layer layer : _layers) {
			layer.afterDecode(p, this);
		}
	}

	public void prepareExpression(ExpressionScope scope) {
		scope.putExpressionElement("thisComp", this);

		for (Layer layer : _layers) {
			layer.prepareExpression(scope.clone());
		}
	}

	public VideoBuffer renderVideoFrame() {
		ColorMode colorMode = getColorMode();
		Size2i size = _vrContext.getRenderResolution().scale(getSize());
		Time time = _vrContext.getTime();

		_vrContext.setColorMode(colorMode);

		// 上のレイヤーから(リスト内では末尾から)順に有効なカメラレイヤーを探し、見つかればそれを使う。
		CameraLayer cameraLayer = null;
		for (ListIterator<Layer> it = _layers.listIterator(_layers.size()); it.hasPrevious(); ) {
			Layer layer = it.previous();
			if (layer instanceof CameraLayer && LayerNature.isVideoEnabled(layer)
					&& !layer.getInPoint().after(time) && layer.getOutPoint().after(time)) {
				cameraLayer = (CameraLayer) layer;
				break;
			}
		}

		final CameraImpl camera = new CameraImpl(
				_vrContext, _vrSupport, cameraLayer, getSize(), size);

		final VideoLayerComposerImpl composer = new VideoLayerComposerImpl(
				_vrContext, _vrSupport, colorMode, size, camera.getPosition());

		for (int i = 0, n = _layers.size(); i < n; ++i) {
			Layer layer = _layers.get(i);

			if (!LayerNature.isVideoEnabled(layer)) {
				continue;
			}

			if (layer.getInPoint().after(time) || !layer.getOutPoint().after(time)) {
				continue;
			}

			if (layer instanceof MediaLayer) {
				final MediaLayer ml = (MediaLayer) layer;

				if (ml.getTrackMatte() != TrackMatte.NONE && i+1 < n) {
					Layer matteLayer = _layers.get(i+1);
					if (matteLayer instanceof MediaLayer) {
						setupTrackMatteRenderer(ml, (MediaLayer) matteLayer, composer, camera, null);
						continue;
					}
				}

				_vrContext.saveAndExecute(new WrappedOperation<Object>() {
					public Object execute() {
						ml.setupVideoRenderer(composer, camera, null);
						return null;
					}
				});
			}
		}

		return composer.compose();
	}

	public void setupCollapseTransformation(
			final VideoLayerComposer composer, final Camera camera, final CollapseTransformation ct) {

		Time time = _vrContext.getTime();

		for (int i = 0, n = _layers.size(); i < n; ++i) {
			Layer layer = _layers.get(i);

			if (!LayerNature.isVideoEnabled(layer)) {
				continue;
			}

			if (layer.getInPoint().after(time) || !layer.getOutPoint().after(time)) {
				continue;
			}

			if (layer instanceof MediaLayer) {
				final MediaLayer ml = (MediaLayer) layer;

				if (ml.getTrackMatte() != TrackMatte.NONE && i+1 < n) {
					Layer matteLayer = _layers.get(i+1);
					if (matteLayer instanceof MediaLayer) {
						setupTrackMatteRenderer(ml, (MediaLayer) matteLayer, composer, camera, ct);
						continue;
					}
				}

				_vrContext.saveAndExecute(new WrappedOperation<Object>() {
					public Object execute() {
						ml.setupVideoRenderer(composer, camera, ct);
						return null;
					}
				});
			}
		}
	}

	private void setupTrackMatteRenderer(
			MediaLayer fillLayer, MediaLayer matteLayer,
			VideoLayerComposer composer, Camera camera, CollapseTransformation ct) {

		final VideoLayerRenderer[] renderers = new VideoLayerRenderer[2];
		final Vec3d[][] vertices = new Vec3d[1][];
		final String[] intersectionGroup = new String[1];

		fillLayer.setupVideoRenderer(new VideoLayerComposer() {
			public void add2D(VideoLayerRenderer r)							{ renderers[0] = r; }
			public void add3D(VideoLayerRenderer r, Vec3d[] v, String ig)	{ renderers[0] = r; vertices[0] = v; intersectionGroup[0] = ig; }
		}, camera, ct);

		matteLayer.setupVideoRenderer(new VideoLayerComposer() {
			public void add2D(VideoLayerRenderer r)							{ renderers[1] = r; }
			public void add3D(VideoLayerRenderer r, Vec3d[] v, String ig)	{ renderers[1] = r; }
		}, camera, ct);


		if (renderers[0] == null) {
			return;

		} else if (renderers[1] == null) {
			if (vertices[0] == null) {
				composer.add2D(renderers[0]);
			} else {
				composer.add3D(renderers[0], vertices[0], intersectionGroup[0]);
			}
			return;
		}


		final TrackMatte trackMatte = fillLayer.getTrackMatte();

		VideoLayerRenderer r = new VideoLayerRenderer() {
			public VideoLayerBuffer render(boolean withDepthBuffer) {
				VideoLayerBuffer fillBuffer = null;
				VideoLayerBuffer matteBuffer = null;
				VideoBuffer vbFill = null;
				try {
					fillBuffer = renderers[0].render(withDepthBuffer);
					matteBuffer = renderers[1].render(false);
					vbFill = fillBuffer.getVideoBuffer();

					VideoBuffer resultBuffer = _vrSupport.createVideoBuffer(vbFill.getColorMode(), vbFill.getBounds());
					_vrSupport.trackMatte(matteBuffer.getVideoBuffer(), fillBuffer.getVideoBuffer(), resultBuffer,
							trackMatte, matteBuffer.getOpacity());

					VideoLayerBuffer newBuffer = new VideoLayerBufferImpl(
							resultBuffer, fillBuffer.getDepthBuffer(), fillBuffer.getBlendMode(), fillBuffer.getOpacity());
					fillBuffer = null;
					return newBuffer;

				} finally {
					if (matteBuffer != null) {
						matteBuffer.dispose();
					}

					// 正常時はfillBufferを破棄してしまうとDepthBufferまで破棄されてしまうので、VideoBufferだけ破棄する。
					// (tryブロックを抜ける直前にfillBufferにnullを代入しているので、ここでfillBufferがnullでないのは例外発生時のみ)
					if (fillBuffer != null) {
						fillBuffer.dispose();
					} else if (vbFill != null) {
						vbFill.dispose();
					}
				}
			}
		};

		if (vertices[0] == null) {
			composer.add2D(r);
		} else {
			composer.add3D(r, vertices[0], intersectionGroup[0]);
		}
	}

	public AudioBuffer renderAudioChunk() {
		AudioBuffer ab = null;
		Time time = _arContext.getTime();
		Time chunkDuration = Time.fromFrameNumber(
				_arContext.getFrameCount(), _arContext.getAudioMode().sampleDuration);

		for (Layer layer : _layers) {
			if (!LayerNature.isAudioEnabled(layer)) {
				continue;
			}

			if (layer.getInPoint().after(time.add(chunkDuration)) || !layer.getOutPoint().after(time)) {
				continue;
			}

			if (layer instanceof MediaLayer) {
				final MediaLayer al = (MediaLayer) layer;

				AudioBuffer alab = _arContext.saveAndExecute(new WrappedOperation<AudioBuffer>() {
					public AudioBuffer execute() {
						return al.renderAudioChunk();
					}
				});

				if (alab != null) {
					if (ab == null) {
						ab = alab;
					} else {
						AudioBuffer abOld = ab;
						try {
							ab = _arSupport.mix(ab, alab);
						} finally {
							if (ab != abOld) {
								abOld.dispose();
							}
							if (ab != alab) {
								alab.dispose();
							}
						}
					}
				}
			}
		}

		return ab;
	}

	public Object createExpressionElement(RenderContext renderContext) {
		return new LayerCompositionExpressionElement(renderContext);
	}

	public class LayerCompositionExpressionElement {

		protected final RenderContext renderContext;

		public LayerCompositionExpressionElement(RenderContext renderContext) {
			this.renderContext = renderContext;
		}

		public Object layer(int index) {
			Layer layer = _layers.get(_layers.size() - index);
			return renderContext.getExpressionElement(layer);
		}

		public int getHeight() {
			return getSize().width;
		}

		public int getWidth() {
			return getSize().height;
		}

	}

}
