/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.CollapseTransformation;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.VideoLayerComposer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.VideoRenderContext;

import com.google.inject.Inject;

@ProjectElement("mediaItemLayer")
public class MediaItemLayerImpl extends AbstractMediaLayer implements MediaItemLayer {

	private String _itemId;

	private MediaItem _mediaItem;

	private boolean _ctcr;

	@Inject
	private VideoRenderContext _vrContext;


	@Override
	protected void initialize(boolean videoAvailable, boolean audioAvailable) {
		throw new UnsupportedOperationException("Use initialize(MediaItem) method instead.");
	}

	public void initialize(MediaItem mediaItem) {
		MediaInput input = mediaItem.getMediaInput();
		boolean videoAvailable = input.isVideoAvailable();
		boolean audioAvailable = input.isAudioAvailable();

		if (!videoAvailable && !audioAvailable) {
			throw new IllegalArgumentException("No video nor audio is available.");
		}

		super.initialize(videoAvailable, audioAvailable);

		_itemId = mediaItem.getId();
		_mediaItem = mediaItem;

		setName(mediaItem.getName());
	}

	public String getItemId() {
		return _itemId;
	}

	public void setItemId(String itemId) {
		_itemId = itemId;
	}

	@JSONHint(ignore=true)
	public MediaItem getItem() {
		return _mediaItem;
	}

	public boolean isCTCR() {
		return _ctcr;
	}

	public void setCTCR(boolean ctcr) {
		_ctcr = ctcr;
	}

	@JSONHint(ignore=true)
	public MediaInput getMediaInput() {
		return _mediaItem.getMediaInput();
	}

	@JSONHint(ignore=true)
	public boolean isPrecompositionLayer() {
		return (getPrecomposition() != null);
	}

	private LayerComposition getPrecomposition() {
		if (_mediaItem instanceof CompositionItem) {
			Composition comp = ((CompositionItem) _mediaItem).getComposition();
			if (comp instanceof LayerComposition) {
				return (LayerComposition) comp;
			}
		}
		return null;
	}

	@Override
	public void afterDecode(Project p, LayerComposition c) throws ProjectDecodeException {
		super.afterDecode(p, c);

		_mediaItem = p.getItem(_itemId);
		if (_mediaItem == null) {
			throw new ProjectDecodeException(
					"no such MediaItem found: id=" + _itemId);
		}
	}

	@Override
	public void prepareExpression(ExpressionScope scope) {
		super.prepareExpression(scope);

		LayerComposition preComp = getPrecomposition();
		if (preComp != null) {
			ExpressionScope preCompScope = scope.createPrecompositionScope(preComp);
			if (preCompScope != null) {
				preComp.prepareExpression(preCompScope);
			}
		}
	}

	@Override
	public void setupVideoRenderer(
			VideoLayerComposer composer, Camera camera, CollapseTransformation ct) {

		if (LayerNature.isCTCR(this)) {
			LayerComposition preComp = getPrecomposition();
			if (preComp != null) {
				setupCollapseTransformation(composer, camera, ct, preComp);
				return;
			}
		}

		super.setupVideoRenderer(composer, camera, ct);
	}

	private void setupCollapseTransformation(
			final VideoLayerComposer composer, final Camera camera,
			final CollapseTransformation ct, final LayerComposition preComp) {

		final double opacity;
		if (ct != null) {
			opacity = getOpacity().value(_vrContext) / 100d * ct.getOpacity();
		} else {
			opacity = getOpacity().value(_vrContext) / 100d;
		}

		final CollapseTransformation ct2 = new CollapseTransformation() {
			public CollapseTransformation getParent()	{ return ct; }
			public MediaItemLayer getLayer()			{ return MediaItemLayerImpl.this; }
			public double getOpacity()					{ return opacity; }
		};

		_vrContext.saveAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				preComp.setupCollapseTransformation(composer, camera, ct2);
				return null;
			}
		});
	}

}
