/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

public abstract class BlueGreenScreenKeyBase {

	private final IVideoEffectContext context;

	private final IShaderProgram keyingProgram;

	private final IShaderProgram maskOnlyProgram;

	private final IShaderProgram smoothingLowProgram;

	private final IShaderProgram smoothingHighProgram;


	protected BlueGreenScreenKeyBase(
			IVideoEffectContext context, IShaderRegistry shaders, boolean blue) {

		this.context = context;

		if (blue) {
			keyingProgram = shaders.getProgram(KeyingShaders.class, "BLUE_SCREEN_KEY");
			maskOnlyProgram = shaders.getProgram(KeyingShaders.class, "BLUE_SCREEN_KEY_MASK_ONLY");
		} else {
			keyingProgram = shaders.getProgram(KeyingShaders.class, "GREEN_SCREEN_KEY");
			maskOnlyProgram = shaders.getProgram(KeyingShaders.class, "GREEN_SCREEN_KEY_MASK_ONLY");
		}

		smoothingLowProgram = shaders.getProgram(KeyingShaders.class, "SMOOTHING_LOW");
		smoothingHighProgram = shaders.getProgram(KeyingShaders.class, "SMOOTHING_HIGH");
	}

	protected IVideoBuffer doKeying(
			IAnimatableDouble threshold,
			IAnimatableDouble cutoff,
			IAnimatableBoolean maskOnly,
			IAnimatableEnum<Smoothing> smoothing) {

		double t = context.value(threshold) / 100;
		double c = context.value(cutoff) / 100;
		boolean mask = context.value(maskOnly);
		Smoothing sm = mask ? Smoothing.NONE : context.value(smoothing);

		double t_minus_c = t - c;
		if (t_minus_c == 0) {
			if (t == 0) {
				t_minus_c = Double.POSITIVE_INFINITY;
			} else if (t == 1) {
				t = Double.POSITIVE_INFINITY;
			}
		}

		IVideoBuffer input = context.doPreviousEffect();
		IVideoBuffer buf = null;
		try {
			VideoBounds bounds = input.getBounds();
			int w = bounds.width;
			int h = bounds.height;

			buf = context.createVideoBuffer(bounds);

			GL gl = context.getGL();
			GLU glu = context.getGLU();

			VideoEffectUtil.ortho2D(gl, glu, w, h);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, buf.getTexture(), 0);
			gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

			gl.glActiveTexture(GL.GL_TEXTURE0);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, input.getTexture());

			IShaderProgram program = mask ? maskOnlyProgram : keyingProgram;
			synchronized (program) {
				gl.glUseProgram(program.getProgram());
				gl.glUniform1i(program.getUniformLocation("texture"), 0);
				gl.glUniform1f(program.getUniformLocation("threshold"), (float) t);
				gl.glUniform1f(program.getUniformLocation("t_minus_c"), (float) t_minus_c);

				gl.glBegin(GL.GL_QUADS);
				gl.glTexCoord2f(0, 0);
				gl.glVertex2f(0, 0);
				gl.glTexCoord2f(w, 0);
				gl.glVertex2f(w, 0);
				gl.glTexCoord2f(w, h);
				gl.glVertex2f(w, h);
				gl.glTexCoord2f(0, h);
				gl.glVertex2f(0, h);
				gl.glEnd();
	
				gl.glFinish();
				gl.glUseProgram(0);
			}

			switch (sm) {
				case LOW:  program = smoothingLowProgram; break;
				case HIGH: program = smoothingHighProgram; break;
				default:   program = null; break;
			}

			if (program != null) {
				IVideoBuffer tmp = input;
				input = buf;
				buf = tmp;

				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, buf.getTexture(), 0);
				gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, input.getTexture());

				synchronized (program) {
					gl.glUseProgram(program.getProgram());
					gl.glUniform1i(program.getUniformLocation("texture"), 0);

					gl.glBegin(GL.GL_QUADS);
					gl.glTexCoord2f(0, 0);
					gl.glVertex2f(0, 0);
					gl.glTexCoord2f(w, 0);
					gl.glVertex2f(w, 0);
					gl.glTexCoord2f(w, h);
					gl.glVertex2f(w, h);
					gl.glTexCoord2f(0, h);
					gl.glVertex2f(0, h);
					gl.glEnd();
		
					gl.glFinish();
					gl.glUseProgram(0);
				}
			}

			gl.glActiveTexture(GL.GL_TEXTURE0);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

			IVideoBuffer output = buf;
			buf = null;
			return output;

		} finally {
			input.dispose();
			if (buf != null) buf.dispose();
		}
	}

}
