/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.program.Program;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.FileItem;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Interpolation;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.SolidColorItem;

public class ImageUtil {

	private static final byte[] COLOR_ICON_BYTES = {
		0, 0, 127, -2, 127, -2, 127, -2, 127, -2, 127, -2, 127, -2, 127, -2,
		127, -2, 127, -2, 127, -2, 127, -2, 127, -2, 127, -2, 127, -2, 0, 0
	};

	private static final RGB RGB_BLACK = new RGB(0, 0, 0);

	private static final Color COLOR_BLACK = new Color(0, 0, 0);


	private static ImageRegistry getRegistry() {
		return Activator.getDefault().getImageRegistry();
	}

	private static Image getColorIcon(Color color) {
		// ここで作成された画像は破棄されることなく保持され続けるが、
		// ピクセルデータ自体は COLOR_ICON_BYTES で共有しており、
		// 画像ひとつ当たりのメモリ消費量は PaletteData オブジェクトの分と
		// ImageRegistry 内のMapの管理領域の分だけなのでまず問題になることはないと思われる。

		ImageRegistry registry = getRegistry();

		final int red = (int)(color.r*255);
		final int green = (int)(color.g*255);
		final int blue = (int)(color.b*255);
		String key = String.format("color.%d.%d.%d", red, green, blue);

		Image image = registry.get(key);
		if (image != null) {
			return image;
		}

		ImageDescriptor descriptor = new ImageDescriptor() {
			public ImageData getImageData() {
				PaletteData palette = new PaletteData(new RGB[] { RGB_BLACK, new RGB(red, green, blue) } );
				return new ImageData(16, 16, 1, palette, 1, COLOR_ICON_BYTES);
			}
		};

		registry.put(key, descriptor);
		return registry.get(key);
	}

	public static Image getSolidColorItemIcon(SolidColorItem item) {
		return getColorIcon(item.getColor());
	}

	public static Image getFileItemIcon(FileItem item) {
		ImageRegistry registry = getRegistry();

		String name = item.getFile().getName();
		int lastDot = name.lastIndexOf('.');
		if (lastDot == -1) {
			return PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJ_FILE);
		}

		String extension = name.substring(lastDot + 1);
		String key = "program." + extension;

		Image image = registry.get(key);
		if (image != null) {
			return image;
		}

		Program program = Program.findProgram(extension);
		if (program == null) {
			return PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJ_FILE);
		}

		@SuppressWarnings("restriction")
		ImageDescriptor descriptor = new org.eclipse.ui.internal.misc.ExternalProgramImageDescriptor(program);
		registry.put(key, descriptor);
		return registry.get(key);
	}

	public static Image getItemIcon(Item item) {
		if (item instanceof Folder) {
			return PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJ_FOLDER);
		}

		if (item instanceof FileItem) {
			return getFileItemIcon((FileItem) item);
		}

		if (item instanceof SolidColorItem) {
			return getSolidColorItemIcon((SolidColorItem) item);
		}

		if (item instanceof CompositionItem) {
			// TODO コンポジション用のアイコン
			return PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJ_ELEMENT);
		}

		// TODO デフォルトのアイコンはもっといいの他にない？
		return PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJ_ELEMENT);
	}

	public static Image getCameraIcon() {
		return getRegistry().get(Activator.IMG_CAMERA);
	}

	public static Image getNullIcon() {
		return getColorIcon(COLOR_BLACK);
	}

	public static Image getTextIcon() {
		return getRegistry().get(Activator.IMG_TEXT);
	}

	public static Image getKeyframeIcon(Interpolation interpolation, boolean selected) {
		String key = "IMG_KF_" + interpolation.name() + (selected ? "_SELECT" : "");
		return getRegistry().get(key);
	}

	public static Image getStopwatchIcon(boolean hasKeyframe) {
		return getRegistry().get(hasKeyframe ? Activator.IMG_STOPWATCH_HAS_KF : Activator.IMG_STOPWATCH_NO_KF);
	}

	public static ImageDescriptor getPlayerZoomIcon() {
		return getRegistry().getDescriptor(Activator.IMG_PLAYER_ZOOM);
	}

	public static ImageDescriptor getPlayerResolutionIcon() {
		return getRegistry().getDescriptor(Activator.IMG_PLAYER_RESOLUTION);
	}

	public static ImageDescriptor getPlayerMuteIcon(boolean muted) {
		return getRegistry().getDescriptor(muted ? Activator.IMG_PLAYER_MUTE : Activator.IMG_PLAYER_NOMUTE);
	}

	public static ImageDescriptor getPlayerInfoIcon() {
		return getRegistry().getDescriptor(Activator.IMG_PLAYER_INFO);
	}

	public static ImageDescriptor getPlayerPinIcon() {
		return getRegistry().getDescriptor(Activator.IMG_PLAYER_PIN);
	}


	private ImageUtil() { }

}
