/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collections;
import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;

public class MoveEffectsOperation extends ProjectOperation {

	private final String _compId;

	private final String _srcLayerId;

	private final String _dstLayerId;

	private final List<Integer> _effectIndices = Util.newList();

	private final int _dstIndex;


	public MoveEffectsOperation(ProjectManager pm,
			EffectableLayer srcLayer, List<Effect> effects,
			EffectableLayer dstLayer, int moveBefore) {

		super(pm, "エフェクトの移動");

		LayerComposition comp = pm.checkLayer(srcLayer);
		if (pm.checkLayer(dstLayer) != comp) {
			throw new IllegalArgumentException();
		}

		_compId = comp.getId();
		_srcLayerId = srcLayer.getId();
		_dstLayerId = dstLayer.getId();

		boolean sameLayer = (srcLayer == dstLayer);

		int dstEffectCount = dstLayer.getEffects().size();
		int dstIndex;
		if (moveBefore < 0 || moveBefore > dstEffectCount) {
			dstIndex = moveBefore = dstEffectCount;
		} else {
			dstIndex = moveBefore;
		}

		for (Effect e : effects) {
			int index = srcLayer.getEffects().indexOf(e);
			if (index == -1) {
				throw new IllegalArgumentException();
			}
			_effectIndices.add(index);
			if (sameLayer && index < moveBefore) {
				--dstIndex;
			}
		}

		_dstIndex = dstIndex;

		// 引数effectsがソートされていない場合はソートしておく必要がある。
		Collections.sort(_effectIndices);

		if (sameLayer) {
			List<Effect> tmp = Util.newList(srcLayer.getEffects());
			tmp.removeAll(effects);
			tmp.addAll(_dstIndex, effects);
			_noEffect = srcLayer.getEffects().equals(tmp);
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);
		EffectableLayer srcLayer = (EffectableLayer) comp.getLayer(_srcLayerId);
		EffectableLayer dstLayer = (EffectableLayer) comp.getLayer(_dstLayerId);

		List<Object[]> data = null;
		if (pm != null) {
			data = Util.newList();
		}

		List<Effect> effects = Util.newList();
		for (int i : _effectIndices) {
			effects.add(srcLayer.getEffects().get(i));
			if (pm != null) {
				data.add(new Object[] { dstLayer, _dstIndex + data.size() });
			}
		}

		srcLayer.getEffects().removeAll(effects);
		if (pm != null) {
			pm.fireEffectsRemove(comp, new Object[][] { new Object[] { srcLayer, -1 } });
		}

		dstLayer.getEffects().addAll(_dstIndex, effects);
		if (pm != null) {
			pm.fireEffectsAdd(comp, data.toArray(new Object[data.size()][]));
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);
		EffectableLayer srcLayer = (EffectableLayer) comp.getLayer(_srcLayerId);
		EffectableLayer dstLayer = (EffectableLayer) comp.getLayer(_dstLayerId);

		List<Object[]> data = null;
		if (pm != null) {
			data = Util.newList();
		}

		List<Effect> effects = Util.newList(
				dstLayer.getEffects().subList(_dstIndex, _dstIndex + _effectIndices.size()));

		dstLayer.getEffects().removeAll(effects);
		if (pm != null) {
			pm.fireEffectsRemove(comp, new Object[][] { new Object[] { dstLayer, -1 } });
		}

		for (int i = 0, n = effects.size(); i < n; ++i) {
			int j = _effectIndices.get(i);
			srcLayer.getEffects().add(j, effects.get(i));
			if (pm != null) {
				data.add(new Object[] { srcLayer, j });
			}
		}
		if (pm != null) {
			pm.fireEffectsAdd(comp, data.toArray(new Object[data.size()][]));
		}

		return Status.OK_STATUS;
	}

}
