/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collections;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectElementFactory;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class NewFolderOperation extends ProjectOperation {

	private final String _folderId;

	private final String _data;

	@Inject
	private ProjectElementFactory _elementFactory;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public NewFolderOperation(ProjectManager projectManager, Folder parent, String name) {
		super(projectManager, "新規フォルダ");
		projectManager.checkItem(parent);

		InjectorHolder.getInjector().injectMembers(this);

		Folder folder = _elementFactory.newFolder();
		folder.setName(name);
		folder.setParent(parent);

		_folderId = folder.getId();
		_data = _encoder.encodeElement(folder);
	}

	public NewFolderOperation(ProjectManager projectManager, Folder parent) {
		this(projectManager, parent, projectManager.getUnusedItemName("名称未設定"));
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		Folder folder = null;

		try {
			folder = _decoder.decodeElement(_data, Folder.class);
			folder.afterDecode(project);
		} catch (ProjectDecodeException e) {
			if (folder != null) {
				folder.dispose();
			}
			throw new ExecutionException("error decoding folder data", e);
		}

		project.getItems().add(folder);

		if (pm != null) {
			fireItemsAdd(folder, pm);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		Folder folder = project.getItem(_folderId);
		project.getItems().remove(folder);

		if (pm != null) {
			fireItemsRemove(folder, pm);
		}

		folder.dispose();
		return Status.OK_STATUS;
	}

	private void fireItemsAdd(Folder folder, ProjectManager pm) {
		Folder parent = folder.getParent();
		if (parent != null) {
			pm.fireItemsAdd(Collections.singleton(folder), Collections.singleton(parent));
		} else {
			pm.fireItemsAdd(Collections.singleton(folder), null);
		}
	}

	private void fireItemsRemove(Folder folder, ProjectManager pm) {
		Folder parent = folder.getParent();
		if (parent != null) {
			pm.fireItemsRemove(Collections.singleton(folder), Collections.singleton(parent));
		} else {
			pm.fireItemsRemove(Collections.singleton(folder), null);
		}
	}

	public String getFolderId() {
		return _folderId;
	}

}
