/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.util.List;
import java.util.Map;

import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Event;

import ch.kuramo.javie.app.Activator;
import ch.kuramo.javie.app.project.ModifyEffectPropertyOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.PropertyDescriptor;
import ch.kuramo.javie.core.Util;

public class EffectElement extends Element {

	public final Effect effect;

	private final List<Element> children = Util.newList();

	private final Map<PropertyDescriptor, Element> childrenMap = Util.newMap();

	private final SwitchGroup enableSwitch = new SwitchGroup();


	public EffectElement(EffectsElement parent, Effect effect) {
		super(parent);
		if (!getLayer().getEffects().contains(effect)) {
			throw new IllegalArgumentException();
		}
		this.effect = effect;

		enableSwitch.add(new EnableSwitch());
	}

	public String getColumnText(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return effect.getName();

			default:
				return null;
		}
	}

	@Override
	public boolean hasChildren() {
		return (effect.getEffectDescriptor().getPropertyDescriptors().size() > 0);
	}

	@Override
	public Element[] getChildren() {
		prepareChildren();
		return children.toArray(new Element[children.size()]);
	}

	@Override
	public void paintColumn(Event event) {
		switch (event.index) {
			case LayerCompositionView.SHOWHIDE_COL:
				enableSwitch.paint(event);
				break;

			default:
				super.paintColumn(event);
				break;
		}
	}

	@Override
	public void mouseDown(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.SHOWHIDE_COL:
				enableSwitch.mouseDown(event);
				break;

			default:
				super.mouseDown(event, columnIndex);
				break;
		}
	}

	private EffectableLayer getLayer() {
		return ((EffectableLayer) ((LayerElement) parent.parent).layer);
	}

	private Element getChildElement(PropertyDescriptor pd) {
		Element child = childrenMap.get(pd);
		if (child == null) {
			if (AnimatableValue.class.isAssignableFrom(pd.getPropertyClass())) {
				child = new EffectAnimatableValueElement(this, getLayer(), effect, pd);
			} else {
				// TODO AnimatableValue以外のプロパティの場合。
				child = new Element(this);
			}
			childrenMap.put(pd, child);
		}
		return child;
	}

	private void prepareChildren() {
		children.clear();

		for (PropertyDescriptor pd : effect.getEffectDescriptor().getPropertyDescriptors()) {
			children.add(getChildElement(pd));
		}

		childrenMap.values().retainAll(children);
	}


	private static final ImageRegistry imageRegistry = Activator.getDefault().getImageRegistry();

	private class EnableSwitch implements Switch {

		public boolean hasBorder() {
			return true;
		}

		public Image getImage() {
			return effect.isEnabled() ? imageRegistry.get(Activator.IMG_SWITCH_EFFECT) : null;
		}

		public void mouseDown(MouseEvent event) {
			boolean currentEnabled = effect.isEnabled();
			String label = currentEnabled ? "エフェクトの停止" : "エフェクトの使用";

			ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
			pm.postOperation(new ModifyEffectPropertyOperation<Boolean>(
					pm, getLayer(), effect, "enabled", !currentEnabled, label));
		}
	}

}
