/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;

import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.app.Activator;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableVec3d;
import ch.kuramo.javie.core.Layer;

public class LayerTranslate2DElement extends LayerAnimatableValueElement {

	public LayerTranslate2DElement(Element parent, Layer layer, String property, String name) {
		super(parent, layer, property, name);
	}

	protected AnimatableValueElementDelegate<?> createDelegate() {
		AnimatableVec3d avalue = PropertyUtil.getProperty(layer, property);
		return new LayerTranslate2DElementDelegate(this, name, avalue);
	}

}

class LayerTranslate2DElementDelegate extends AnimatableVec3dElementDelegate {

	private static final Image translate2DImage = Activator.getDefault().getImageRegistry().get(Activator.IMG_SWITCH_TRANSLATE2D);


	private final SwitchGroup translate2DSwitch = new SwitchGroup();


	public LayerTranslate2DElementDelegate(
			AnimatableValueElement element, String name, AnimatableVec3d avalue) {

		super(element, name, avalue, false);

		translate2DSwitch.add(new Switch() {
			public void mouseDown(MouseEvent event) {
				if (dragGestureEditor == null) {
					originalValue = value;
					dragGestureEditor = new Translate2DDragGestureEditor(event);
				}
			}

			public boolean hasBorder() {
				return false;
			}

			public Image getImage() {
				return translate2DImage;
			}
		});
	}

	@Override
	public void paintColumn(Event event) {
		if (event.index == LayerCompositionView.VALUE_COL) {
			translate2DSwitch.paint(event);
		}
		super.paintColumn(event);
	}

	@Override
	protected void drawValue(GC gc, int x, int y, int height, boolean focused) {
		super.drawValue(gc, x+18, y, height, focused);
	}

	@Override
	protected int drawValue(GC gc, int x, int y, int height, int valueIndex, double value) {
		return super.drawValue(gc, (valueIndex == 0) ? x+18 : x, y, height, valueIndex, value);
	}

	@Override
	public void mouseDown(MouseEvent event, int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			translate2DSwitch.mouseDown(event);
		}
		super.mouseDown(event, columnIndex);
	}

	@Override
	public CellEditor getCellEditor(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			if (editor == null) {
				editor = new Translate2DCellEditor(element.viewer.getTree(), SWT.SINGLE | SWT.BORDER);
			}
			return editor;
		}
		return null;
	}

	@Override
	protected Vec3d dragGesture(double dx, double dy) {
		if (dragGestureEditor instanceof Translate2DDragGestureEditor) {
			return new Vec3d(originalValue.x + increment*dx, originalValue.y + increment*dy, value.z);
		} else {
			return super.dragGesture(dx, dy);
		}
	}

	protected class Translate2DCellEditor extends Vec3dCellEditor {

		protected Translate2DCellEditor(Composite parent, int style) {
			super(parent, style);
		}

		@Override
		protected Control createControl(Composite parent) {
			Control control = super.createControl(parent);

			FormData data = (FormData) text.getLayoutData();
			data.left.offset += 18;
			data.right.offset += 18;

			if (WIN32) {
				control.addPaintListener(new PaintListener() {
					public void paintControl(PaintEvent e) {
						Event event = new Event();
						event.gc = e.gc;
						event.widget = e.widget;
						event.x = e.x;
						event.y = e.y;
						translate2DSwitch.paint(event);
					}
				});
			}

			return control;
		}

	}

	// translate2DSwitchで発生したジェスチャーかどうかを識別するためだけにオーバーライドしている。
	protected class Translate2DDragGestureEditor extends DragGestureEditor {
		protected Translate2DDragGestureEditor(MouseEvent event) {
			super(event);
		}
	}

}
