/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.io.File;

import net.arnx.jsonic.JSONHint;

import org.mozilla.javascript.Scriptable;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.AnimatableColor;
import ch.kuramo.javie.core.AnimatableInteger;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.DepthBuffer;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.RenderContext;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.VectorMediaInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.exprelems.StringProperty;
import ch.kuramo.javie.core.services.FontManager;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

import ftgl.FTGL;
import ftgl.FTGLfont;

@ProjectElement("textLayer")
public class TextLayerImpl extends AbstractMediaLayer implements TextLayer {

	private final TextInput _input = new TextInput();

	private boolean _ctcr;

	private AnimatableString _sourceText = new AnimatableString("テキスト");

	private AnimatableInteger _fontSize = new AnimatableInteger(36);

	private AnimatableColor _fillColor = new AnimatableColor(new Color(1, 1, 1));

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private FontManager _fontManager;


	@Override
	protected void initialize(boolean videoAvailable, boolean audioAvailable) {
		throw new UnsupportedOperationException("Use initialize() method instead.");
	}

	@Override
	public void initialize() {
		super.initialize(true, false);
	}

	@JSONHint(ignore=true)
	public MediaInput getMediaInput() {
		return _input;
	}

	public boolean isCTCR() {
		return _ctcr;
	}

	public void setCTCR(boolean ctcr) {
		_ctcr = ctcr;
	}

	public AnimatableString getSourceText() {
		return _sourceText;
	}

	public void setSourceText(AnimatableString sourceText) {
		_sourceText = sourceText;
	}

	public AnimatableInteger getFontSize() {
		return _fontSize;
	}

	public void setFontSize(AnimatableInteger fontSize) {
		_fontSize = fontSize;
	}

	public AnimatableColor getFillColor() {
		return _fillColor;
	}

	public void setFillColor(AnimatableColor fillColor) {
		_fillColor = fillColor;
	}

	@Override
	public void prepareExpression(ExpressionScope scope) {
		super.prepareExpression(scope);
		scope.assignTo(_sourceText);
		scope.assignTo(_fontSize);
		scope.assignTo(_fillColor);
	}

	@Override
	public Object createExpressionElement(RenderContext renderContext) {
		return new TextLayerExpressionElement(renderContext);
	}

	public class TextLayerExpressionElement extends MediaLayerExpressionElement {

		public TextLayerExpressionElement(RenderContext renderContext) {
			super(renderContext);
		}

		public StringProperty getSourceText()	{ return elem(_sourceText); }
		public Scriptable getFontSize()			{ return elem(_fontSize); }
		public Scriptable[] getFillColor()		{ return elem(_fillColor); }
	}

	private class TextInput implements VectorMediaInput {

		public void dispose() {
			// nothing to do
		}

		public boolean isVideoAvailable() {
			return true;
		}
		
		public boolean isAudioAvailable() {
			return false;
		}

		public Time getDuration() {
			return null;
		}

		public Time getVideoFrameDuration() {
			return null;
		}

		public VideoBounds getVideoFrameBounds() {
			if (!_vrContext.isActive()) {
				return null;
			}

			String sourceText = _sourceText.value(_vrContext);

			// TODO サイズ0のVideoBoundsを返すとエラーになるので
			//      暫定的な対処として (1, 1) を返しているが、もっと適切な方法を考える。
			//      (プレビュー解像度がFULL以外の場合はこの対処ではエラーになるなど、不完全)
			if (sourceText.length() == 0) {
				return new VideoBounds(1, 1);
			}

			// [left, lower, near, right, upper, far]
			float[] box = new float[6];
			FTGL.instance.ftglGetFontBBox(getFont(), sourceText, -1, box);

			return new VideoBounds(box[0], -box[4],
					(int) Math.ceil(box[3]-box[0]), (int) Math.ceil(box[4]-box[1]));
		}

		public VideoBuffer getVideoFrameImage() {
			VideoBounds bounds = _vrContext.getRenderResolution().scale(getVideoFrameBounds());
			String sourceText = _sourceText.value(_vrContext);
			Color fillColor = _fillColor.value(_vrContext);

			VideoBuffer vb = _vrSupport.createVideoBuffer(_vrContext.getColorMode(), bounds);
			vb.allocateAsTexture();
			vb.clear();

			_vrSupport.renderText(getFont(), fillColor, sourceText, vb);

			return vb;
		}

		public void rasterize(
				VideoBuffer resultBuffer, DepthBuffer depthBuffer,
				double[] mvMatrix, double[] prjMatrix) {

			String sourceText = _sourceText.value(_vrContext);
			Color fillColor = _fillColor.value(_vrContext);

			_vrSupport.renderText(
					getFont(), fillColor, sourceText, resultBuffer, depthBuffer, mvMatrix, prjMatrix);
		}

		public AudioBuffer getAudioChunk() {
			throw new UnsupportedOperationException("audio is not available");
		}

		private FTGLfont getFont() {
			int fontSize = _fontSize.value(_vrContext);

			if (_ctcr) {
				return _fontManager.getPolygonFont(FONTPATH, fontSize);
			} else {
				return _fontManager.getBufferFont(FONTPATH, fontSize);
			}
		}

	}


	// TODO フォントを選択できるようにする。以下は暫定。

	private static final String FONTPATH;

	static {
		String[] fonts = {
				"C:\\WINDOWS\\Fonts\\meiryob.ttc",
				"C:\\WINDOWS\\Fonts\\msgoth04.ttc",
				"C:\\WINDOWS\\Fonts\\msgothic.ttc",
				"/System/Library/Fonts/ヒラギノ角ゴ ProN W6.otf",
				"/System/Library/Fonts/ヒラギノ角ゴ Pro W6.otf",
				"/Library/Fonts/Osaka.dfont"
		};

		String fontpath = "/no/font/available.ttc";
		for (String f : fonts) {
			if (new File(f).isFile()) {
				fontpath = f;
				break;
			}
		}
		FONTPATH = fontpath;
	}

}
