/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.plugin.PIVideoBuffer;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public class VideoBufferImpl implements VideoBuffer, PIVideoBuffer {

	private static final Logger _logger = LoggerFactory.getLogger(VideoBufferImpl.class);

	@Inject
	private VideoRenderSupport _vrSupport;

	private final ColorMode _colorMode;

	private final Size2i _imageSize;

	private final VideoBounds _bounds;

	private int _texture;

	private Object _array;


	public VideoBufferImpl(ColorMode colorMode, Size2i imageSize) {
		_colorMode = colorMode;
		_imageSize = imageSize;
		_bounds = new VideoBounds(imageSize.width, imageSize.height);
	}

	public VideoBufferImpl(ColorMode colorMode, VideoBounds bounds) {
		_colorMode = colorMode;
		_imageSize = new Size2i(bounds.width, bounds.height);
		_bounds = bounds;
	}

	@Override
	protected void finalize() throws Throwable {
		if (_texture != 0 || _array != null) {
			// TODO 解放忘れの場合どうする？
			//		finalize は別のスレッドから呼ばれるので、ここでテクスチャの解放をすることはできない。
			//		生成時に VideoRenderContext に登録しておいて、
			//		VideoRenderContext の deactivate で検出／解放するといいかも。
			_logger.warn("finalizing a VideoBufferImpl object, but the object is not disposed.");

			if (_array != null) {
				_vrSupport.releaseArray(_array);
				_array = null;
			}
		}

		super.finalize();
	}

	public void dispose() {
		if (_texture != 0) {
			_vrSupport.deleteTexture(_texture);
			_texture = 0;
		}

		if (_array != null) {
			_vrSupport.releaseArray(_array);
			_array = null;
		}
	}

	public void allocateAsTexture() {
		allocate(true);
	}

	public void allocateAsArray() {
		allocate(false);
	}

	private void allocate(boolean asTexture) {
		if (_texture != 0 || _array != null) {
			throw new IllegalStateException("already allocated");
		}

		if (asTexture) {
			getTexture();
		} else {
			getArray();
		}
	}

	public boolean isAllocated() {
		return (isTexture() || isArray());
	}

	public boolean isTexture() {
		return (_texture != 0);
	}

	public boolean isArray() {
		return (_array != null);
	}

	public void clear() {
		_vrSupport.clear(this);
	}

	public int getTexture() {
		if (_texture == 0) {
			_texture = _vrSupport.createTexture(_colorMode, _imageSize);

		} else if (_array != null) {
			// bug?
			throw new IllegalStateException("both texture and array exist.");
		}

		// 配列にデータがある場合、テクスチャにコピーして配列を解放する。
		if (_array != null) {
			_vrSupport.copyArrayToTexture(_array, _texture, _colorMode, _imageSize);
			_vrSupport.releaseArray(_array);
			_array = null;
		}

		return _texture;
	}

	public Object getArray() {
		if (_array == null) {
			_array = _vrSupport.createArray(_colorMode, _imageSize);

		} else if (_texture != 0) {
			// bug?
			throw new IllegalStateException("both texture and array exist.");
		}

		// テクスチャにデータがある場合、配列にコピーしてテクスチャを破棄する
		if (_texture != 0) {
			_vrSupport.copyTextureToArray(_texture, _array, _colorMode, _imageSize);
			_vrSupport.deleteTexture(_texture);
			_texture = 0;
		}

		return _array;
	}

	public ColorMode getColorMode() {
		return _colorMode;
	}

	public Size2i getImageSize() {
		return _imageSize;
	}

	public VideoBounds getBounds() {
		return _bounds;
	}

}
