/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.LinkedList;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.plugin.PIAnimatableValue;
import ch.kuramo.javie.api.plugin.PIAudioBuffer;
import ch.kuramo.javie.api.plugin.PIAudioRenderContext;
import ch.kuramo.javie.core.Util;

public class PIAudioRenderContextProxy implements PIAudioRenderContext {

	private static final ThreadLocal<LinkedList<PIAudioRenderContext>>
			_threadLocal = new ThreadLocal<LinkedList<PIAudioRenderContext>>();

	static void push(PIAudioRenderContext impl) {
		getStack(true).addLast(impl);
	}

	static void pop() {
		LinkedList<PIAudioRenderContext> stack = getStack(false);

		stack.removeLast();

		if (stack.isEmpty()) {
			_threadLocal.remove();
		}
	}

	private static LinkedList<PIAudioRenderContext> getStack(boolean create) {
		LinkedList<PIAudioRenderContext> stack = _threadLocal.get();
		if (stack == null) {
			if (create) {
				stack = Util.newLinkedList();
				_threadLocal.set(stack);
			} else {
				throw new IllegalStateException("no stack is found");
			}
		} else if (stack.isEmpty()) {
			// popメソッドにおいて、スタックが空になったときにはスレッドローカルからスタックを取り除いているので、
			// ここで空のスタックが返されることは無い。
			throw new IllegalStateException("stack is empty");
		}
		return stack;
	}

	private PIAudioRenderContext getImpl() {
		return getStack(false).getLast();
	}

	public PIAudioBuffer doPreviousEffect() {
		return getImpl().doPreviousEffect();
	}

	public Time getTime() {
		return getImpl().getTime();
	}

	public void setTime(Time time) {
		getImpl().setTime(time);
	}

	public <V> V value(PIAnimatableValue<V> avalue) {
		return getImpl().value(avalue);
	}

	public AudioMode getAudioMode() {
		return getImpl().getAudioMode();
	}

	public int getFrameCount() {
		return getImpl().getFrameCount();
	}

	public void setFrameCount(int count) {
		getImpl().setFrameCount(count);
	}

	public PIAudioBuffer createAudioBuffer() {
		return getImpl().createAudioBuffer();
	}

}
