/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "QTCoreVideoInputJNI.h"
#include "QTCoreVideoInput.h"


#define JLONG(p) ((jlong) (pointer_t) p)
#define POINTER(type, p) ((type) (pointer_t) p)


JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM* vm, void* reserved)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	
	NSUserDefaults* defaults = [NSUserDefaults standardUserDefaults];
	[defaults setInteger:300*60*60 forKey:@"QuartzComposerDefaultMovieDuration"];
	
	[pool release];
	return JNI_VERSION_1_4;
}

NSString* jstringToNSString(JNIEnv* env, jstring jstr)
{
	const jchar* chars = (*env)->GetStringChars(env, jstr, NULL);
	NSString* nsstr = [NSString stringWithCharacters:(UniChar*)chars
											  length:(*env)->GetStringLength(env, jstr)];
	(*env)->ReleaseStringChars(env, jstr, chars);
	return nsstr;
}

JNIEXPORT jlongArray JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTCoreVideoInput_openVideo
(JNIEnv* env, jobject jthis, jstring jfilename, jlong pixelFormat)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	
	EnterMoviesOnThread(0);
	CSSetComponentsThreadMode(kCSAcceptAllComponentsMode);
	
	VideoInput* videoInput = [[VideoInput alloc] initWithFile:jstringToNSString(env, jfilename)
												   cglContext:CGLGetCurrentContext()
											   cglPixelFormat:[POINTER(NSOpenGLPixelFormat*, pixelFormat) CGLPixelFormatObj]];
	
	jlongArray result = NULL;
	
	if (videoInput != NULL) {
		jlong buf[6] = { JLONG(videoInput), [videoInput width], [videoInput height],
						[videoInput sampleCount], [videoInput duration], [videoInput timeScale] };
		result = (*env)->NewLongArray(env, 6);
		(*env)->SetLongArrayRegion(env, result, 0, 6, buf);
	}
	
	[pool release];
	return result;
}

JNIEXPORT jlongArray JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTCoreVideoInput_openAudio
(JNIEnv* env, jobject jthis, jstring jfilename)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	
	EnterMoviesOnThread(0);
	CSSetComponentsThreadMode(kCSAcceptAllComponentsMode);
	
	AudioInput* audioInput = [[AudioInput alloc] initWithFile:jstringToNSString(env, jfilename)];
	
	jlongArray result = NULL;
	
	if (audioInput != NULL) {
		jlong buf[4] = { JLONG(audioInput), [audioInput sampleCount], [audioInput duration], [audioInput timeScale] };
		result = (*env)->NewLongArray(env, 4);
		(*env)->SetLongArrayRegion(env, result, 0, 4, buf);
	}
	
	[pool release];
	return result;
}

JNIEXPORT void JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTCoreVideoInput_closeVideo
(JNIEnv* env, jobject jthis, jlong videoInputAddress)
{
	VideoInput* videoInput = POINTER(VideoInput*, videoInputAddress);
	[videoInput release];
	ExitMoviesOnThread();
}

JNIEXPORT void JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTCoreVideoInput_closeAudio
(JNIEnv* env, jobject jthis, jlong audioInputAddress)
{
	AudioInput* audioInput = POINTER(AudioInput*, audioInputAddress);
	[audioInput release];
	ExitMoviesOnThread();
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTCoreVideoInput_frameImageAtTime
(JNIEnv* env, jobject jthis, jlong videoInputAddress,
 jlong timeValue, jint timeScale, jint texture, jint width, jint height)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	
	VideoInput* videoInput = POINTER(VideoInput*, videoInputAddress);
	jint error = [videoInput frameImageAtTime:QTMakeTime(timeValue, timeScale)
									  texture:texture width:width height:height];
	
	[pool release];
	return error;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTCoreVideoInput_setAudioDescription
(JNIEnv* env, jobject jthis, jlong audioInputAddress,
 jint channels, jint sampleRate, jint sampleSize, jboolean floatingPoint)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	
	NSDictionary* description = [NSDictionary dictionaryWithObjectsAndKeys:
								 [NSNumber numberWithInt:channels], @"channels",
								 [NSNumber numberWithInt:sampleRate], @"sampleRate",
								 [NSNumber numberWithInt:sampleSize], @"sampleSize",
								 [NSNumber numberWithBool:floatingPoint], @"floatingPoint",
								 NULL];
	
	AudioInput* audioInput = POINTER(AudioInput*, audioInputAddress);
	jint error = [audioInput setAudioDescription:description];
	
	[pool release];
	return error;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_MacOSXQTCoreVideoInput_audioChunkFromTime
(JNIEnv* env, jobject jthis, jlong audioInputAddress,
 jlong timeValue, jint timeScale, jobject byteArray, jint frameCount)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	
	void* buffer = (*env)->GetPrimitiveArrayCritical(env, byteArray, NULL);
	
	AudioInput* audioInput = POINTER(AudioInput*, audioInputAddress);
	jint error = [audioInput audioChunkFromTime:QTMakeTime(timeValue, timeScale)
										 buffer:buffer frameCount:frameCount];
	
	(*env)->ReleasePrimitiveArrayCritical(env, byteArray, buffer, 0);
	
	[pool release];
	return error;
}
