/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.GLProgram;
import ch.kuramo.javie.api.annotations.GLShader;
import ch.kuramo.javie.api.annotations.GLShader.ShaderType;
import ch.kuramo.javie.api.plugin.PIShaderProgram;
import ch.kuramo.javie.api.plugin.PIShaderRegistry;
import ch.kuramo.javie.api.plugin.PIVideoBuffer;

public class VideoEffectUtil {

	public static void ortho2D(GL gl, GLU glu, int width, int height) {
		gl.glViewport(0, 0, width, height);

		gl.glMatrixMode(GL.GL_PROJECTION);
		gl.glLoadIdentity();
		glu.gluOrtho2D(0, width, 0, height);

		gl.glMatrixMode(GL.GL_MODELVIEW);
		gl.glLoadIdentity();
	}

	private static final String[] createConvolution1DSource(boolean h) {
		return new String[] {
				"uniform sampler2DRect texture;",
				"uniform int radius;",
				"uniform float kernel[101];",
				"",
				"void main(void)",
				"{",
				"	vec2 texCoord = gl_TexCoord[0].st;",
				"	vec4 sum = vec4(0.0);",
				"	for (int i = -radius; i <= radius; ++i) {",
			h ?	"		vec4 src = texture2DRect(texture, vec2(texCoord.s+float(i), texCoord.t));"
			  :	"		vec4 src = texture2DRect(texture, vec2(texCoord.s, texCoord.t+float(i)));",
				"		sum += kernel[radius+i] * src;",
				"	}",
				"	gl_FragColor = sum;",
				"}"
		};		
	}

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram
	public static final String[] CONVOLUTION_1D_H = createConvolution1DSource(true);

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram
	public static final String[] CONVOLUTION_1D_V = createConvolution1DSource(false);


	public static void convolution1D(
			PIVideoBuffer src, PIVideoBuffer dst,
			boolean horizontal, int radius, float[] kernel,
			GL gl, GLU glu, PIShaderRegistry shaders) {

		VideoBounds srcBounds = src.getBounds();
		VideoBounds dstBounds = dst.getBounds();
		float x1 = (float) (dstBounds.x - srcBounds.x);
		float y1 = (float) (dstBounds.y - srcBounds.y);
		float x2 = x1 + dstBounds.width;
		float y2 = y1 + dstBounds.height;

		ortho2D(gl, glu, dstBounds.width, dstBounds.height);
		gl.glTranslatef(-x1, -y1, 0);

		gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
				GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, dst.getTexture(), 0);
		gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

		gl.glActiveTexture(GL.GL_TEXTURE0);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, src.getTexture());

		PIShaderProgram program = shaders.getProgram(
				VideoEffectUtil.class, horizontal ? "CONVOLUTION_1D_H" : "CONVOLUTION_1D_V");
		synchronized (program) {
			gl.glUseProgram(program.getProgram());
			gl.glUniform1i(program.getUniformLocation("texture"), 0);
			gl.glUniform1i(program.getUniformLocation("radius"), radius);
			gl.glUniform1fv(program.getUniformLocation("kernel[0]"), radius*2+1, kernel, 0);

			gl.glBegin(GL.GL_QUADS);
			gl.glTexCoord2f(x1, y1);
			gl.glVertex2f(x1, y1);
			gl.glTexCoord2f(x2, y1);
			gl.glVertex2f(x2, y1);
			gl.glTexCoord2f(x2, y2);
			gl.glVertex2f(x2, y2);
			gl.glTexCoord2f(x1, y2);
			gl.glVertex2f(x1, y2);
			gl.glEnd();

			gl.glFinish();
			gl.glUseProgram(0);
		}

		gl.glActiveTexture(GL.GL_TEXTURE0);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

		gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
				GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);
	}

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram
	public static final String[] CONVOLUTION = {
		"uniform sampler2DRect texture;",
		"uniform int ksize;",
		"uniform float kernel[49];",
		"uniform vec2 offset[49];",
		"",
		"void main(void)",
		"{",
		"	vec2 texCoord = gl_TexCoord[0].st;",
		"	vec4 sum = vec4(0.0);",
		"	for (int i = 0; i < ksize; ++i) {",
		"		vec4 src = texture2DRect(texture, texCoord + offset[i]);",
		"		sum += kernel[i] * src;",
		"	}",
		"	gl_FragColor = sum;",
		"}"
	};

	public static void convolution(
			PIVideoBuffer src, PIVideoBuffer dst,
			int ksize, float[] kernel, float[] offset,
			GL gl, GLU glu, PIShaderRegistry shaders) {

		VideoBounds srcBounds = src.getBounds();
		VideoBounds dstBounds = dst.getBounds();
		float x1 = (float) (dstBounds.x - srcBounds.x);
		float y1 = (float) (dstBounds.y - srcBounds.y);
		float x2 = x1 + dstBounds.width;
		float y2 = y1 + dstBounds.height;

		ortho2D(gl, glu, dstBounds.width, dstBounds.height);
		gl.glTranslatef(-x1, -y1, 0);

		gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
				GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, dst.getTexture(), 0);
		gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

		gl.glActiveTexture(GL.GL_TEXTURE0);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, src.getTexture());

		PIShaderProgram program = shaders.getProgram(VideoEffectUtil.class, "CONVOLUTION");
		synchronized (program) {
			gl.glUseProgram(program.getProgram());
			gl.glUniform1i(program.getUniformLocation("texture"), 0);
			gl.glUniform1i(program.getUniformLocation("ksize"), ksize);
			gl.glUniform1fv(program.getUniformLocation("kernel[0]"), ksize, kernel, 0);
			gl.glUniform2fv(program.getUniformLocation("offset[0]"), ksize*2, offset, 0);

			gl.glBegin(GL.GL_QUADS);
			gl.glTexCoord2f(x1, y1);
			gl.glVertex2f(x1, y1);
			gl.glTexCoord2f(x2, y1);
			gl.glVertex2f(x2, y1);
			gl.glTexCoord2f(x2, y2);
			gl.glVertex2f(x2, y2);
			gl.glTexCoord2f(x1, y2);
			gl.glVertex2f(x1, y2);
			gl.glEnd();

			gl.glFinish();
			gl.glUseProgram(0);
		}

		gl.glActiveTexture(GL.GL_TEXTURE0);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

		gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
				GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);
	}

	public static VideoBounds expandBounds(VideoBounds bounds, int expand, boolean h, boolean v) {
		int dx = 0, dy = 0, dw = 0, dh = 0;

		if (h) {
			dx = -expand;
			dw = expand*2;
		}

		if (v) {
			dy = -expand;
			dh = expand*2;
		}

		return new VideoBounds(
				bounds.x + dx, bounds.y + dy,
				bounds.width + dw, bounds.height + dh);
	}

	public static void setNearest(PIVideoBuffer vb, GL gl) {
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, vb.getTexture());
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_MIN_FILTER, GL.GL_NEAREST);
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_MAG_FILTER, GL.GL_NEAREST);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
	}

	public static void setLinear(PIVideoBuffer vb, GL gl) {
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, vb.getTexture());
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_MIN_FILTER, GL.GL_LINEAR);
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_MAG_FILTER, GL.GL_LINEAR);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
	}

	public static void setClampToEdge(PIVideoBuffer vb, GL gl) {
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, vb.getTexture());
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_WRAP_S, GL.GL_CLAMP_TO_EDGE);
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_WRAP_T, GL.GL_CLAMP_TO_EDGE);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
	}

	public static void setClampToBorder(PIVideoBuffer vb, GL gl) {
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, vb.getTexture());
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_WRAP_S, GL.GL_CLAMP_TO_BORDER);
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_WRAP_T, GL.GL_CLAMP_TO_BORDER);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
	}

	public static void clearTexture(PIVideoBuffer vb, GL gl) {
		if (vb.isArray()) {
			throw new IllegalArgumentException();
		}

		gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
				GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, vb.getTexture(), 0);
		gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

		gl.glClearColor(0, 0, 0, 0);
		gl.glClear(GL.GL_COLOR_BUFFER_BIT);

		gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
				GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);
	}


	private VideoEffectUtil() { }

}
