/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.distort;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.GLProgram;
import ch.kuramo.javie.api.annotations.GLShader;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.GLShader.ShaderType;
import ch.kuramo.javie.api.plugin.PIAnimatableDouble;
import ch.kuramo.javie.api.plugin.PIAnimatableEnum;
import ch.kuramo.javie.api.plugin.PIShaderProgram;
import ch.kuramo.javie.api.plugin.PIShaderRegistry;
import ch.kuramo.javie.api.plugin.PIVideoBuffer;
import ch.kuramo.javie.api.plugin.PIVideoRenderContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.PolarCoordinates", category=Effect.DISTORT)
public class PolarCoordinates {

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram
	public static final String[] RECT_TO_POLAR = {
		"uniform sampler2DRect texture;",
		"uniform vec2 center;",
		"uniform float radians_per_px;",
		"uniform float interp;",
		"",
		"void main(void)",
		"{",
		"	float sRect = gl_TexCoord[0].s;",
		"	float tRect = gl_TexCoord[0].t;",
		"",
		"	float sPolar = atan(sRect-center.s, tRect-center.t) / radians_per_px + center.s;",
		"	float tPolar = distance(gl_TexCoord[0].st, center) * 2.0;",
		"",
		"	float s = interp*sPolar + (1.0-interp)*sRect;",
		"	float t = interp*tPolar + (1.0-interp)*tRect;",
		"	gl_FragColor = texture2DRect(texture, vec2(s, t));",
		"}"
	};

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram
	public static final String[] POLAR_TO_RECT = {
		"uniform sampler2DRect texture;",
		"uniform vec2 center;",
		"uniform float radians_per_px;",
		"uniform float maxRadius_div_h;",
		"uniform float interp;",
		"",
		"void main(void)",
		"{",
		"	float sPolar = gl_TexCoord[0].s;",
		"	float tPolar = gl_TexCoord[0].t;",
		"",
		"	float theta = (sPolar - center.s) * radians_per_px;",
		"	float sRect = maxRadius_div_h*tPolar*cos(theta) + center.s;",
		"	float tRect = maxRadius_div_h*tPolar*sin(theta) + center.t;",
		"",
		"	float s = interp*sRect + (1.0-interp)*sPolar;",
		"	float t = interp*tRect + (1.0-interp)*tPolar;",
		"	gl_FragColor = texture2DRect(texture, vec2(s, t));",
		"}"
	};


	public enum TypeOfConversion { RECT_TO_POLAR, POLAR_TO_RECT }


	private final PIVideoRenderContext context;

	private final PIShaderProgram rectToPolarProgram;

	private final PIShaderProgram polarToRectProgram;

	@Property
	private PIAnimatableDouble interpolation;

	@Property
	private PIAnimatableEnum<TypeOfConversion> typeOfConversion;


	@Inject
	public PolarCoordinates(PIVideoRenderContext context, PIShaderRegistry shaders) {
		this.context = context;
		rectToPolarProgram = shaders.getProgram(PolarCoordinates.class, "RECT_TO_POLAR");
		polarToRectProgram = shaders.getProgram(PolarCoordinates.class, "POLAR_TO_RECT");
	}

	public PIVideoBuffer doVideoEffect() {
		// TODO 値の範囲制限は @Property アノテーションで行う。
		double interp = Math.max(0, Math.min(1, context.value(this.interpolation)/100));
		if (interp == 0) {
			return null;
		}

		TypeOfConversion type = context.value(this.typeOfConversion);
		PIShaderProgram program = (type == TypeOfConversion.RECT_TO_POLAR)
				? rectToPolarProgram : polarToRectProgram;

		PIVideoBuffer input = context.doPreviousEffect();
		PIVideoBuffer buffer = null;
		try {
			VideoBounds bounds = input.getBounds();
			int w = bounds.width;
			int h = bounds.height;

			buffer = context.createVideoBuffer(bounds);

			GL gl = context.getGL();
			GLU glu = context.getGLU();

			VideoEffectUtil.ortho2D(gl, glu, w, h);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, buffer.getTexture(), 0);
			gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

			gl.glActiveTexture(GL.GL_TEXTURE0);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, input.getTexture());

			synchronized (program) {
				gl.glUseProgram(program.getProgram());
				gl.glUniform1i(program.getUniformLocation("texture"), 0);
				gl.glUniform2f(program.getUniformLocation("center"), w/2f, h/2f);
				gl.glUniform1f(program.getUniformLocation("radians_per_px"), (float)Math.toRadians(360.0/w));
				if (type == TypeOfConversion.POLAR_TO_RECT) {
					gl.glUniform1f(program.getUniformLocation("maxRadius_div_h"), (float)(Math.sqrt(w*w+h*h)*0.5/h));
				}
				gl.glUniform1f(program.getUniformLocation("interp"), (float)interp);

				gl.glBegin(GL.GL_QUADS);
				gl.glTexCoord2f(0, 0);
				gl.glVertex2f(0, 0);
				gl.glTexCoord2f(w, 0);
				gl.glVertex2f(w, 0);
				gl.glTexCoord2f(w, h);
				gl.glVertex2f(w, h);
				gl.glTexCoord2f(0, h);
				gl.glVertex2f(0, h);
				gl.glEnd();

				gl.glFinish();
				gl.glUseProgram(0);
			}

			gl.glActiveTexture(GL.GL_TEXTURE0);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

			PIVideoBuffer output = buffer;
			buffer = null;
			return output;

		} finally {
			input.dispose();
			if (buffer != null) buffer.dispose();
		}
	}

}
