/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.transition;

import java.nio.FloatBuffer;
import java.util.Arrays;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.plugin.PIAnimatableDouble;
import ch.kuramo.javie.api.plugin.PIArrayPools;
import ch.kuramo.javie.api.plugin.PIVideoBuffer;
import ch.kuramo.javie.api.plugin.PIVideoRenderContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.LinearWipe", category=Effect.TRANSITION)
public class LinearWipe {

	private final PIVideoRenderContext context;

	private final PIArrayPools arrayPools;

	@Property
	private PIAnimatableDouble transitionCompletion;

	@Property("90")
	private PIAnimatableDouble wipeAngle;

	@Property
	private PIAnimatableDouble feather;


	@Inject
	public LinearWipe(PIVideoRenderContext context, PIArrayPools arrayPools) {
		this.context = context;
		this.arrayPools = arrayPools;
	}

	public PIVideoBuffer doVideoEffect() {
		// TODO 値の範囲制限は @Property アノテーションで行う。
		double completion = Math.max(0, Math.min(1, context.value(transitionCompletion)/100));
		if (completion == 0) {
			return null;
		} else if (completion == 1) {
			// TODO context.doPreviousEffect()をしなくてもVideoBoundsを取得できる方法が必要。
			//      context.getPreviousBounds()をここで呼ぶとIllegalStateExceptionが発生する。
			PIVideoBuffer vb = context.doPreviousEffect();
			VideoEffectUtil.clearTexture(vb, context.getGL());
			return vb;
		}

		double wipeAngle = context.value(this.wipeAngle) % 360;
		if (wipeAngle < 0) wipeAngle += 360;

		// TODO 値の範囲制限は @Property アノテーションで行う。
		double feather = Math.max(0, context.value(this.feather));
		feather = context.getRenderResolution().scale(feather);


		PIVideoBuffer input = context.doPreviousEffect();
		VideoBounds bounds = input.getBounds();
		int w = bounds.width;
		int h = bounds.height;

		double theta = Math.toRadians(wipeAngle);
		double wipeLen = h*Math.abs(Math.cos(theta))+w*Math.abs(Math.sin(theta));
		double wipeWidth = w*Math.abs(Math.cos(theta))+h*Math.abs(Math.sin(theta));

		GL gl = context.getGL();
		GLU glu = context.getGLU();

		VideoEffectUtil.ortho2D(gl, glu, w, h);

		gl.glMatrixMode(GL.GL_MODELVIEW);
		gl.glTranslatef(w/2f, h/2f, 0);
		gl.glRotatef((float)wipeAngle, 0, 0, 1);
		gl.glScalef((float)(wipeWidth/w), (float)(wipeLen/h), 1);
		gl.glTranslatef(-w/2f, -h/2f, 0);

		int[] wipeTex = new int[1];
		gl.glGenTextures(1, wipeTex, 0);
		try {
			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, input.getTexture(), 0);
			gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

			gl.glActiveTexture(GL.GL_TEXTURE0);
			gl.glBindTexture(GL.GL_TEXTURE_1D, wipeTex[0]);
			gl.glTexParameteri(GL.GL_TEXTURE_1D, GL.GL_TEXTURE_MIN_FILTER, GL.GL_LINEAR);
			gl.glTexParameteri(GL.GL_TEXTURE_1D, GL.GL_TEXTURE_MAG_FILTER, GL.GL_LINEAR);
			gl.glTexParameteri(GL.GL_TEXTURE_1D, GL.GL_TEXTURE_WRAP_S, GL.GL_CLAMP_TO_EDGE);
			gl.glTexParameteri(GL.GL_TEXTURE_1D, GL.GL_TEXTURE_WRAP_T, GL.GL_CLAMP_TO_EDGE);
			prepareWipeTexture((int)Math.round(wipeLen), (float)completion, (float)feather, gl);
			gl.glEnable(GL.GL_TEXTURE_1D);

			gl.glEnable(GL.GL_BLEND);
			gl.glBlendFuncSeparate(GL.GL_ZERO, GL.GL_SRC_ALPHA, GL.GL_ZERO, GL.GL_SRC_ALPHA);

			gl.glBegin(GL.GL_QUADS);
			gl.glTexCoord1f(1);
			gl.glVertex2f(0, 0);
			gl.glTexCoord1f(1);
			gl.glVertex2f(w, 0);
			gl.glTexCoord1f(0);
			gl.glVertex2f(w, h);
			gl.glTexCoord1f(0);
			gl.glVertex2f(0, h);
			gl.glEnd();

			gl.glDisable(GL.GL_BLEND);
			gl.glDisable(GL.GL_TEXTURE_1D);
			gl.glBindTexture(GL.GL_TEXTURE_1D, 0);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);
		} finally {
			gl.glDeleteTextures(1, wipeTex, 0);
		}

		return input;
	}

	private void prepareWipeTexture(int wipeLen, float completion, float feather, GL gl) {
		int intFeather = (int) Math.ceil(feather);
		int tmpDataLen = wipeLen + 4*intFeather;

		float completionPoint = intFeather + (wipeLen+2*intFeather) * completion;
		int completionIndex = (int) completionPoint;

		float[] tmpData = arrayPools.getFloatArray(tmpDataLen);
		float[] wipeData = null;
		float[] kernel = null;

		try {
			FloatBuffer buffer;
			if (completionIndex+intFeather == tmpDataLen) {
				// ArrayPoolsから取得した配列はゼロクリアされていないのでクリアする必要がある。
				Arrays.fill(tmpData, 0);
				buffer = FloatBuffer.wrap(tmpData);

			} else if (tmpDataLen == wipeLen) {
				Arrays.fill(tmpData, 0, completionIndex, 0f);
				tmpData[completionIndex] = 1 - (completionPoint - completionIndex);
				Arrays.fill(tmpData, completionIndex+1, tmpDataLen, 1f);
				buffer = FloatBuffer.wrap(tmpData);

			} else {
				int kernelLen = intFeather*2+1;
				kernel = arrayPools.getFloatArray(kernelLen);

				float sigma = feather / 2.5f;
				float sigmaSquare = sigma * sigma;
				float ksum = 0;
				for (int i = 1; i <= intFeather; ++i) {
					ksum += 2 * (kernel[intFeather-i] = (float) Math.exp(-i * i / (2 * sigmaSquare)));
				}
				kernel[intFeather] = 1 / (++ksum);
				for (int i = 1; i <= intFeather; ++i) {
					kernel[intFeather+i] = (kernel[intFeather-i] /= ksum);
				}

				float fractionAtCompletionIndex = 1 - (completionPoint - completionIndex);

				Arrays.fill(tmpData, 0, completionIndex-intFeather, 0f);
				tmpData[completionIndex-intFeather] = fractionAtCompletionIndex*kernel[0];

				float sum = 0;
				for (int k = 1; k < kernelLen; ++k) {
					sum += kernel[k];
					tmpData[completionIndex-intFeather+k] = sum + fractionAtCompletionIndex*kernel[k-1];
				}

				Arrays.fill(tmpData, completionIndex+intFeather+1, tmpDataLen, 1f);

				wipeData = arrayPools.getFloatArray(wipeLen);
				System.arraycopy(tmpData, 2*intFeather, wipeData, 0, wipeLen);
				buffer = FloatBuffer.wrap(wipeData);
			}

			switch (context.getColorMode()) {
				case RGBA8:
					gl.glTexImage1D(GL.GL_TEXTURE_1D, 0, GL.GL_ALPHA8, wipeLen, 0, GL.GL_ALPHA, GL.GL_FLOAT, buffer);
					break;

				case RGBA16:
					gl.glTexImage1D(GL.GL_TEXTURE_1D, 0, GL.GL_ALPHA16, wipeLen, 0, GL.GL_ALPHA, GL.GL_FLOAT, buffer);
					break;

				case RGBA16_FLOAT:
					gl.glTexImage1D(GL.GL_TEXTURE_1D, 0, GL.GL_ALPHA16F_ARB, wipeLen, 0, GL.GL_ALPHA, GL.GL_FLOAT, buffer);
					break;

				case RGBA32_FLOAT:
					gl.glTexImage1D(GL.GL_TEXTURE_1D, 0, GL.GL_ALPHA32F_ARB, wipeLen, 0, GL.GL_ALPHA, GL.GL_FLOAT, buffer);
					break;

				default:
					throw new RuntimeException("unknown ColorMode: " + context.getColorMode());
			}
		} finally {
			arrayPools.put(tmpData);
			arrayPools.put(wipeData);
			arrayPools.put(kernel);
		}
	}

}
