/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.ICoolBarManager;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.action.ToolBarContributionItem;
import org.eclipse.jface.action.ToolBarManager;
import org.eclipse.swt.SWT;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.RetargetAction;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;
import org.eclipse.ui.application.ActionBarAdvisor;
import org.eclipse.ui.application.IActionBarConfigurer;

import ch.kuramo.javie.app.actions.NewCompositionAction;
import ch.kuramo.javie.app.actions.NewFileItemsAction;
import ch.kuramo.javie.app.actions.NewFolderAction;
import ch.kuramo.javie.app.actions.NewProjectAction;
import ch.kuramo.javie.app.actions.NewSolidColorItemAction;
import ch.kuramo.javie.app.actions.OpenProjectAction;
import ch.kuramo.javie.app.actions.RedoAction;
import ch.kuramo.javie.app.actions.SaveAction;
import ch.kuramo.javie.app.actions.SaveAsAction;
import ch.kuramo.javie.app.actions.SaveCopyAction;
import ch.kuramo.javie.app.actions.ShowConsoleViewAction;
import ch.kuramo.javie.app.actions.ShowProjectViewAction;
import ch.kuramo.javie.app.actions.UndoAction;
import ch.kuramo.javie.app.player.GLCanvasFactory;
import ch.kuramo.javie.core.EffectDescriptor;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.EffectRegistry;

import com.google.inject.Inject;

public class ApplicationActionBarAdvisor extends ActionBarAdvisor {

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");


	private NewProjectAction newProjectAction;

	private NewCompositionAction newCompositionAction;

	private NewSolidColorItemAction newSolidColorItemAction;

	private NewFolderAction newFolderAction;

	private OpenProjectAction openProjectAction;

	private SaveAction saveAction;

	private SaveAsAction saveAsAction;

	private SaveCopyAction saveCopyAction;

	private NewFileItemsAction newFileItemsAction;

	private RetargetAction sequenceOutputAction;

	private RetargetAction waveOutputAction;

	private IWorkbenchAction quitAction;

	private UndoAction undoAction;

	private RedoAction redoAction;

	private RetargetAction newCameraLayerAction;

	private RetargetAction newNullLayerAction;

	private RetargetAction newTextLayerAction;

	private IWorkbenchAction newWindowAction;

	private ShowProjectViewAction showProjectViewAction;

	private ShowConsoleViewAction showConsoleViewAction;

	private IWorkbenchAction aboutAction;

	@Inject
	private EffectRegistry effectRegistry;

	private final List<MenuManager> effectCategorySubMenus = Util.newList();

	private final List<Action> defaultCategoryEffectActions = Util.newList();


	public ApplicationActionBarAdvisor(IActionBarConfigurer configurer) {
		super(configurer);
		InjectorHolder.getInjector().injectMembers(this);
	}

	protected void makeActions(IWorkbenchWindow window) {
		newProjectAction = new NewProjectAction(window);
		register(newProjectAction);

		newCompositionAction = new NewCompositionAction(window);
		register(newCompositionAction);

		newSolidColorItemAction = new NewSolidColorItemAction(window);
		register(newSolidColorItemAction);

		newFolderAction = new NewFolderAction(window);
		register(newFolderAction);

		openProjectAction = new OpenProjectAction(window);
		register(openProjectAction);

		saveAction = new SaveAction(window);
		register(saveAction);

		saveAsAction = new SaveAsAction(window);
		register(saveAsAction);

		saveCopyAction = new SaveCopyAction(window);
		register(saveCopyAction);

		newFileItemsAction = new NewFileItemsAction(window);
		register(newFileItemsAction);

		sequenceOutputAction = new RetargetAction(CommandIds.SEQUENCE_OUTPUT, "イメージシーケンス...");
		register(sequenceOutputAction);
		window.getPartService().addPartListener(sequenceOutputAction);

		waveOutputAction = new RetargetAction(CommandIds.WAVE_OUTPUT, "Wave...");
		register(waveOutputAction);
		window.getPartService().addPartListener(waveOutputAction);

		if (!COCOA) {
			quitAction = ActionFactory.QUIT.create(window);
			register(quitAction);
		}

		undoAction = new UndoAction(window);
		register(undoAction);

		redoAction = new RedoAction(window);
		register(redoAction);

		newCameraLayerAction = new RetargetAction(CommandIds.NEW_CAMERA_LAYER, "カメラ");
		register(newCameraLayerAction);
		window.getPartService().addPartListener(newCameraLayerAction);

		newNullLayerAction = new RetargetAction(CommandIds.NEW_NULL_LAYER, "ヌルオブジェクト");
		register(newNullLayerAction);
		window.getPartService().addPartListener(newNullLayerAction);

		newTextLayerAction = new RetargetAction(CommandIds.NEW_TEXT_LAYER, "テキスト");
		register(newTextLayerAction);
		window.getPartService().addPartListener(newTextLayerAction);

		GLCanvasFactory glCanavsFactory = GLCanvasFactory.getFactory();
		if (!glCanavsFactory.isPoolMode()) {
			newWindowAction = ActionFactory.OPEN_NEW_WINDOW.create(window);
			register(newWindowAction);
		}

		showProjectViewAction = new ShowProjectViewAction(window);
		register(showProjectViewAction);

		showConsoleViewAction = new ShowConsoleViewAction(window);
		register(showConsoleViewAction);

		aboutAction = ActionFactory.ABOUT.create(window);
		register(aboutAction);


		makeEffectActions(window);
	}

	private void makeEffectActions(IWorkbenchWindow window) {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint("ch.kuramo.javie.api.effectCategory");
		if (point == null) {
			return;
		}

		Map<String, MenuManager> subMenus = Util.newMap();

		for (IExtension ext : point.getExtensions()) {
			for (IConfigurationElement cfgElem : ext.getConfigurationElements()) {
				if ("effect-category".equals(cfgElem.getName())) {
					String category = cfgElem.getAttribute("category");
					String name = cfgElem.getAttribute("name");

					MenuManager subMenu = new MenuManager(name);
					subMenus.put(category, subMenu);
					effectCategorySubMenus.add(subMenu);
				}
			}
		}

		for (EffectDescriptor ed : effectRegistry.getEffectDescriptors()) {
			RetargetAction action = new RetargetAction("EFFECT." + ed.getType(), ed.getLabel());
			//register(action);
			window.getPartService().addPartListener(action);

			MenuManager subMenu = subMenus.get(ed.getCategory());
			if (subMenu != null) {
				subMenu.add(action);
			} else {
				defaultCategoryEffectActions.add(action);
			}
		}
	}

	protected void fillMenuBar(IMenuManager menuBar) {
		MenuManager fileMenu = new MenuManager("&File", IWorkbenchActionConstants.M_FILE);
		MenuManager fileNewMenu = new MenuManager("新規");
		MenuManager inputMenu = new MenuManager("読み込み");
		MenuManager outputMenu = new MenuManager("書き出し");
		MenuManager editMenu = new MenuManager("&Edit", IWorkbenchActionConstants.M_EDIT);
		MenuManager layerMenu = new MenuManager("&Layer" /* TODO idが必要？ */);
		MenuManager layerNewMenu = new MenuManager("新規");
		MenuManager effectMenu = new MenuManager("&Effect" /* TODO idが必要？ */);
		MenuManager windowMenu = new MenuManager("&Window", IWorkbenchActionConstants.M_WINDOW);
		MenuManager helpMenu = new MenuManager("&Help", IWorkbenchActionConstants.M_HELP);

		menuBar.add(fileMenu);
		menuBar.add(editMenu);
		menuBar.add(layerMenu);
		menuBar.add(effectMenu);
		menuBar.add(windowMenu);
		menuBar.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
		menuBar.add(helpMenu);

		// File
		fileMenu.add(fileNewMenu);
		fileMenu.add(openProjectAction);
		fileMenu.add(new Separator());
		fileMenu.add(saveAction);
		fileMenu.add(saveAsAction);
		fileMenu.add(saveCopyAction);
		fileMenu.add(new Separator());
		fileMenu.add(inputMenu);
		fileMenu.add(outputMenu);
		if (quitAction != null) {
			fileMenu.add(new Separator());
			fileMenu.add(quitAction);
		}

		// File/New
		fileNewMenu.add(newProjectAction);
		fileNewMenu.add(newCompositionAction);
		fileNewMenu.add(newSolidColorItemAction);
		fileNewMenu.add(newFolderAction);

		// File/Input
		inputMenu.add(newFileItemsAction);

		// File/Output
		outputMenu.add(sequenceOutputAction);
		outputMenu.add(waveOutputAction);

		// Edit
		editMenu.add(undoAction);
		editMenu.add(redoAction);

		// Layer
		layerMenu.add(layerNewMenu);

		// Layer/New
		layerNewMenu.add(newCameraLayerAction);
		layerNewMenu.add(newNullLayerAction);
		layerNewMenu.add(newTextLayerAction);

		// Effect
		for (MenuManager effectCategoryMenu : effectCategorySubMenus) {
			effectMenu.add(effectCategoryMenu);
		}
		for (Action effectAction : defaultCategoryEffectActions) {
			effectMenu.add(effectAction);
		}

		// Window
		if (newWindowAction != null) {
			windowMenu.add(newWindowAction);
			windowMenu.add(new Separator());
		}
		windowMenu.add(showProjectViewAction);
		windowMenu.add(showConsoleViewAction);

		// Help
		helpMenu.add(aboutAction);
	}

	protected void fillCoolBar(ICoolBarManager coolBar) {
		IToolBarManager toolBar = new ToolBarManager(SWT.FLAT | SWT.RIGHT);
		coolBar.add(new ToolBarContributionItem(toolBar, "main"));

		GLCanvasFactory glCanavsFactory = GLCanvasFactory.getFactory();
		if (glCanavsFactory.isPoolMode()) {
			toolBar.add(glCanavsFactory.createControlContribution());
		}
	}

}
