/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import ch.kuramo.javie.core.JavieRuntimeException;

public class PropertyUtil {

	public static <T> T getProperty(Object object, String property) {
		Class<?> clazz = object.getClass();
		String upperCamelCase = Character.toUpperCase(property.charAt(0)) + property.substring(1);

		Method method = null;
		try {
			method = clazz.getMethod("get" + upperCamelCase);
			if (method.getReturnType().equals(void.class)) {
				method = null;
			}
		} catch (NoSuchMethodException e) {
		}

		if (method == null) {
			try {
				method = clazz.getMethod("is" + upperCamelCase);
				if (!method.getReturnType().equals(boolean.class)) {
					method = null;
				}
			} catch (NoSuchMethodException e) {
			}
		}

		if (method == null) {
			throw new IllegalArgumentException(String.format(
					"no such property found: class=%s, property=%s",
					clazz.getName(), property));
		}

		try {
			@SuppressWarnings("unchecked")
			T value = (T) method.invoke(object);
			return value;
		} catch (IllegalAccessException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		}
	}

	public static <T> void setProperty(Object object, String property, T newValue) {
		Class<?> clazz = object.getClass();
		String methodName = "set" + Character.toUpperCase(property.charAt(0)) + property.substring(1);

		Method method = null;
		for (Method m : clazz.getMethods()) {
			if (m.getName().equals(methodName) && m.getParameterTypes().length == 1
					&& m.getReturnType().equals(void.class)) {

				Class<?> paramType = toWrapperType(m.getParameterTypes()[0]);
				if (paramType.isAssignableFrom(newValue.getClass())) {
					method = m;
					break;
				}
			}
		}

		if (method == null) {
			throw new IllegalArgumentException(String.format(
					"no such property found: class=%s, property=%s",
					clazz.getName(), property));
		}

		try {
			method.invoke(object, newValue);
		} catch (IllegalAccessException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private static Class<?> toWrapperType(Class<?> type) {
		return type.equals(Boolean.TYPE)   ? Boolean.class
			 : type.equals(Character.TYPE) ? Character.class
			 : type.equals(Byte.TYPE)      ? Byte.class
			 : type.equals(Short.TYPE)     ? Short.class
			 : type.equals(Integer.TYPE)   ? Integer.class
			 : type.equals(Long.TYPE)      ? Long.class
			 : type.equals(Float.TYPE)     ? Float.class
			 : type.equals(Double.TYPE)    ? Double.class
			 : type.equals(Void.TYPE)      ? Void.class
			 : type;
	}

}
