/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.lang.reflect.InvocationTargetException;
import java.nio.ByteBuffer;
import java.util.Arrays;

import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.ui.IWorkbenchWindow;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.CommandIds;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.services.ArrayPools;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class WaveOutputAction extends Action {

	private static File defaultFolder;


	private final LayerCompositionView view;

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;

	@Inject
	private AudioRenderContext arContext;

	@Inject
	private ArrayPools _arrayPools;


	public WaveOutputAction(LayerCompositionView view) {
		super("Wave...");
		InjectorHolder.getInjector().injectMembers(this);

		this.view = view;

		setId(CommandIds.WAVE_OUTPUT);
		setActionDefinitionId(CommandIds.WAVE_OUTPUT);
		//setImageDescriptor(Activator.getImageDescriptor("/icons/wave_output.png"));
	}

	public void run() {
		ProjectManager pm = ProjectManager.forWorkbenchWindow(getWindow());
		if (pm == null) {
			return;
		}

		if (defaultFolder == null) {
			File file = pm.getFile();
			if (file != null) {
				defaultFolder = file.getParentFile();
			}
		}

		CompositionItem compItem = view.getCompositionItem();
		File file = showSaveDialog(defaultFolder, compItem.getName() + ".wav");
		if (file != null) {
			defaultFolder = file.getParentFile();
			doOutput(pm.getProject(), compItem.getId(), file);
		}
	}

	private IWorkbenchWindow getWindow() {
		return view.getSite().getWorkbenchWindow();
	}

	private File showSaveDialog(File folder, String name) {
		String[] filterNames = new String[] { "Wave Files", "All Files (*)" };
		String[] filterExtensions = new String[] { "*.wav", "*" };

		String platform = SWT.getPlatform();
		if (platform.equals("win32") || platform.equals("wpf")) {
			filterNames = new String[] { "Wave Files", "All Files (*.*)" };
			filterExtensions = new String[] { "*.wav", "*.*" };
		}

		FileDialog dialog = new FileDialog(getWindow().getShell(), SWT.SAVE | SWT.SHEET);
		dialog.setFilterNames(filterNames);
		dialog.setFilterExtensions(filterExtensions);
		dialog.setFilterPath(folder != null ? folder.getAbsolutePath() : null);
		dialog.setFileName(name);
		dialog.setOverwrite(true);

		String path = dialog.open();
		return (path != null) ? new File(path) : null;
	}

	private void doOutput(Project project, String compItemId, File file) {
		Project copy = null;
		try {
			copy = decoder.decodeElement(encoder.encodeElement(project), Project.class);
			copy.afterDecode();

			ProgressMonitorDialog dialog = new ProgressMonitorDialog(getWindow().getShell());
			dialog.create();
			dialog.getShell().setText("書き出し");
			dialog.run(true, true, new WaveOutput((CompositionItem) copy.getItem(compItemId), file));

		} catch (ProjectDecodeException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		} catch (InterruptedException e) {
			// ユーザーがキャンセルした場合
			file.delete();
		} finally {
			if (copy != null) {
				// TODO Project#dispose メソッドを作る。
				for (Item i : copy.getItems()) {
					i.dispose();
				}
			}
		}
	}

	private class WaveOutput implements  IRunnableWithProgress {

		private final CompositionItem compItem;

		private final File file;


		private WaveOutput(CompositionItem compItem, File file) {
			this.compItem = compItem;
			this.file = file;
		}

		public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
			Composition comp = compItem.getComposition();
			Time duration = comp.getDuration();
			Time videoFrameDuration = comp.getFrameDuration();
			long videoFrameCount = duration.toFrameNumber(videoFrameDuration);

			monitor.beginTask(String.format("書き出し: %s", compItem.getName()), (int) videoFrameCount);

			AudioMode audioMode = AudioMode.STEREO_48KHZ_INT16;		// TODO 指定できるようにする。
			int chunkFrames = audioMode.sampleRate / 10;			// TODO 1回あたりの処理量はこれで妥当？
			long totalFrames = duration.toFrameNumber(audioMode.sampleDuration);

			PipedOutputStream pout = new PipedOutputStream();
			WaveOutputThread thread;
			try {
				thread = new WaveOutputThread(pout, audioMode, totalFrames, file);
				thread.start();
			} catch (IOException e) {
				throw new InvocationTargetException(e);
			}

			arContext.activate();
			try {
				long videoFrameNumber = 0;
				for (long i = 0; i < totalFrames; ) {
					if (monitor.isCanceled()) {
						throw new InterruptedException();
					} else if (thread.exception != null) {
						throw new InvocationTargetException(thread.exception);
					}

					monitor.subTask(String.format("フレーム: %d/%d", videoFrameNumber, videoFrameCount));

					Time time = Time.fromFrameNumber(i, audioMode.sampleDuration);
					arContext.reset();
					arContext.setAudioMode(audioMode);
					arContext.setTime(time);
					arContext.setFrameCount((int) Math.min(chunkFrames, totalFrames-i));

					AudioBuffer ab = comp.renderAudioChunk();
					try {
						if (ab != null) {
							write(ab, pout);
							i+= ab.getFrameCount();
						} else {
							writeZero(audioMode, arContext.getFrameCount(), pout);
							i += arContext.getFrameCount();
						}
					} catch (IOException e) {
						throw new InvocationTargetException(e);
					} finally {
						if (ab != null) ab.dispose();
					}

					long nextVideoFrameNumber = time.toFrameNumber(videoFrameDuration);
					if (nextVideoFrameNumber != videoFrameNumber) {
						monitor.worked((int)(nextVideoFrameNumber - videoFrameNumber));
						videoFrameNumber = nextVideoFrameNumber;
					}
				}
			} finally {
				arContext.deactivate();
				try {
					pout.close();
				} catch (IOException e) {
					throw new InvocationTargetException(e);
				}
			}

			monitor.done();
		}

		private void write(AudioBuffer ab, OutputStream out) throws IOException {
			AudioMode audioMode = ab.getAudioMode();
			int dataLen = ab.getDataLength();
			int dataLenInBytes = ab.getDataLengthInBytes();
			byte[] outBuf = _arrayPools.getByteArray(dataLenInBytes);

			switch (audioMode.dataType) {
				case SHORT:
					ByteBuffer.wrap(outBuf).asShortBuffer().put((short[]) ab.getData(), 0, dataLen);
					break;
				case INT: {
					int[] data = (int[]) ab.getData();
					for (int i = 0; i < dataLen; ++i) {
						outBuf[i*2  ] = (byte) ((data[i] >>> 24) & 0xff);
						outBuf[i*2+1] = (byte) ((data[i] >>> 16) & 0xff);
					}
					break;
				}
				case FLOAT: {
					float[] data = (float[]) ab.getData();
					for (int i = 0; i < dataLen; ++i) {
						short shortVal = (short) (Math.min(Math.max(data[i], -1.0), 1.0) * Short.MAX_VALUE);
						outBuf[i*2  ] = (byte) ((shortVal >>> 8) & 0xff);
						outBuf[i*2+1] = (byte) ((shortVal      ) & 0xff);
					}
					break;
				}
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}

			out.write(outBuf, 0, dataLenInBytes);
			_arrayPools.put(outBuf);
		}

		private void writeZero(AudioMode audioMode, int frameCount, OutputStream out) throws IOException {
			int dataLenInBytes = audioMode.frameSize * frameCount;
			byte[] outBuf = _arrayPools.getByteArray(dataLenInBytes);
			Arrays.fill(outBuf, (byte) 0);
			out.write(outBuf, 0, dataLenInBytes);
			_arrayPools.put(outBuf);
		}

	}

	private class WaveOutputThread extends Thread {

		final PipedInputStream inputStream;

		final AudioMode audioMode;

		final long totalFrames;

		final File file;

		Exception exception;


		WaveOutputThread(PipedOutputStream src, AudioMode audioMode, long totalFrames, File file) throws IOException {
			inputStream = new PipedInputStream(src);
			this.audioMode = audioMode;
			this.totalFrames = totalFrames;
			this.file = file;
		}

		public void run() {
			AudioFormat format = new AudioFormat(audioMode.sampleRate, audioMode.sampleSize*8, audioMode.channels, true, true);
			AudioInputStream ais = new AudioInputStream(inputStream, format, audioMode.frameSize*totalFrames);
			try {
				AudioSystem.write(ais, AudioFileFormat.Type.WAVE, file);
			} catch (IOException e) {
				exception = e;
			} finally {
				try {
					ais.close();
				} catch (IOException e) {
					exception = e;
				}
			}
		}

	}

}
