/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.player;

import java.awt.Dimension;
import java.awt.EventQueue;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicReference;

import javax.media.opengl.DebugGL;
import javax.media.opengl.GL;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.GLCanvas;
import javax.media.opengl.GLContext;
import javax.media.opengl.GLEventListener;
import javax.media.opengl.glu.GLU;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.ScrollBar;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.player.GLCanvasFactory.GLCanvasRecord;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.FrameDuration;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.services.VideoRenderContext;

import com.google.inject.Inject;

public class VideoCanvas extends Thread implements GLEventListener {

	private static final Logger _logger = LoggerFactory.getLogger(VideoCanvas.class);

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	private static final Time TIME0 = Time.fromFrameNumber(0, FrameDuration.FPS_29_97);


	private final MediaItem _mediaItem;

	private final MediaInput _mediaInput;

	private final TimeKeeper _timeKeeper;

	private GLCanvasRecord _glCanvasRecord;

	private final ScrolledComposite _scrolled;

	private final GLCanvas _canvas;

	private final GLU _glu = new GLU();

	private final AtomicReference<VideoBuffer> _videoBufferRef = new AtomicReference<VideoBuffer>();

	private final Queue<VideoBuffer> _oldVideoBuffers = new ConcurrentLinkedQueue<VideoBuffer>();

	private volatile Time _frameDuration;

	private volatile RenderResolution _resolution = RenderResolution.FULL;

	private double _zoom = Double.NaN;

	private Size2i _canvasSize;

	private int _hScroll;

	private int _vScroll;

	private int _scrollAreaHeight;

	private volatile int[] _viewport = new int[4];

	private final Object _monitor = new Object();

	private boolean _glInitialized;

	private volatile boolean _playing;

	private volatile boolean _forceRender;

	private volatile boolean _finished;

	private volatile boolean _displaying;

	private final ListenerList _playerThreadListeners = new ListenerList();

	@Inject
	private VideoRenderContext _vrContext;


	public VideoCanvas(Composite parent, MediaItem mediaItem, TimeKeeper timeKeeper) {
		super("VideoCanvas: " + mediaItem.getName());

		MediaInput mediaInput = mediaItem.getMediaInput();
		if (mediaInput == null) {
			throw new IllegalArgumentException("no MediaInput is available");
		}
		if (!mediaInput.isVideoAvailable()) {
			throw new IllegalArgumentException("no video is available");
		}

		InjectorHolder.getInjector().injectMembers(this);

		_mediaItem = mediaItem;
		_mediaInput = mediaInput;
		_timeKeeper = timeKeeper;

		_glCanvasRecord = GLCanvasFactory.getFactory().getGLCanvas(parent);
		_scrolled = _glCanvasRecord.scrolled;
		_canvas = _glCanvasRecord.glCanvas;

		_canvas.addGLEventListener(this);

		if (COCOA) {
			listenToScrollAndResize();
			_scrollAreaHeight = _scrolled.getClientArea().height;
		}

		setZoom(1.0);
	}

	public Control getControl() {
		return _scrolled;
	}

	private void cleanup() {
		_canvas.removeGLEventListener(this);

		if (_glCanvasRecord != null) {
			GLCanvasFactory.getFactory().releaseGLCanvas(_glCanvasRecord);
			_glCanvasRecord = null;
		}
	}

	private void listenToScrollAndResize() {
		final ScrollBar hBar = _scrolled.getHorizontalBar();
		final ScrollBar vBar = _scrolled.getVerticalBar();

		hBar.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				int scroll = hBar.getSelection();
				if (_hScroll != scroll) {
					_hScroll = scroll;
					updateViewport();
				}
			}
		});

		vBar.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				int scroll = vBar.getSelection();
				if (_vScroll != scroll) {
					_vScroll = scroll;
					updateViewport();
				}
			}
		});

		_scrolled.addControlListener(new ControlAdapter() {
			public void controlResized(ControlEvent e) {
				if (Double.isNaN(_zoom)) {
					fitCanvasToScrollArea();
				}

				e.display.asyncExec(new Runnable() {
					public void run() {
						if (_scrolled.isDisposed()) {
							return;
						}
						int height = _scrolled.getClientArea().height;
						int hScroll = hBar.getSelection();
						int vScroll = vBar.getSelection();
						if (_scrollAreaHeight != height || _hScroll != hScroll || _vScroll != vScroll) {
							_scrollAreaHeight = height;
							_hScroll = hScroll;
							_vScroll = vScroll;
							updateViewport();
						}
					}
				});
			}
		});
	}

	public void setFrameDuration(Time frameDuration) {
		_frameDuration = frameDuration;
	}

	public void setResolution(RenderResolution resolution) {
		_resolution = resolution;
	}

	public void setZoom(double zoom) {
		if (_zoom != zoom) {
			_zoom = zoom;

			if (Double.isNaN(zoom)) {
				_scrolled.setMinSize(1, 1);
				fitCanvasToScrollArea();

			} else {
				VideoBounds bounds = _mediaInput.getVideoFrameBounds();
				int w = (int) (bounds.width*zoom);
				int h = (int) (bounds.height*zoom);

				_scrolled.setMinSize(w, h);
				setCanvasSize(w, h);
			}
		}
	}

	private void fitCanvasToScrollArea() {
		Rectangle scrollArea = _scrolled.getClientArea();
		VideoBounds videoBounds = _mediaInput.getVideoFrameBounds();

		int w, h;
		double scrollAspect = (double) scrollArea.width / scrollArea.height;
		double videoAspect = (double) videoBounds.width / videoBounds.height;
		if (scrollAspect > videoAspect) {
			w = (int) (scrollArea.height * videoAspect);
			h = scrollArea.height;
		} else {
			w = scrollArea.width;
			h = (int) (scrollArea.width / videoAspect);
		}

		setCanvasSize(w, h);
	}

	private void setCanvasSize(final int width, final int height) {
		_canvasSize = new Size2i(width, height);
		updateViewport();

		EventQueue.invokeLater(new Runnable() {
			public void run() {
				Dimension size = _canvas.getPreferredSize();
				if (size.width != width || size.height != height) {
					_canvas.setPreferredSize(new Dimension(width, height));
					_canvas.invalidate();
					_canvas.getParent().validate();
				}
			}
		});
	}

	private void updateViewport() {
		int[] viewport = new int[] { 0, 0, _canvasSize.width, _canvasSize.height };

		if (COCOA) {
			viewport[0] = -_hScroll;
			if (_scrollAreaHeight < _canvasSize.height) {
				viewport[1] = _vScroll - _canvasSize.height + _scrollAreaHeight;
			}
		}

		if (!Arrays.equals(_viewport, viewport)) {
			_viewport = viewport;
			_canvas.repaint();
		}
	}

	public void init(GLAutoDrawable drawable) {
		GL gl = drawable.getGL();

		gl.glEnable(GL.GL_TEXTURE_RECTANGLE_EXT);
		gl.glColor4f(1, 1, 1, 1);

		// 背景を黒以外にする場合はアルファブレンドを有効にし、blendFuncを次のように設定する。
		//gl.glEnable(GL.GL_BLEND);
		//gl.glBlendFunc(GL.GL_ONE, GL.GL_ONE_MINUS_SRC_ALPHA);

		gl.glClearColor(0, 0, 0, 1);
		gl.glClear(GL.GL_COLOR_BUFFER_BIT);

		gl.glMatrixMode(GL.GL_MODELVIEW);
		gl.glLoadIdentity();

		_glInitialized = true;
	}

	public void display(GLAutoDrawable drawable) {
		if (_finished) {
			return;
		}

		if (!_glInitialized) {
			init(drawable);
		}

		_displaying = true;

		VideoBuffer vb = null;
		try {
			GL gl = drawable.getGL();

			// 背景を黒以外にする場合に必要。initメソッド内のコメント参照。
			// gl.glClear(GL.GL_COLOR_BUFFER_BIT);

			vb = _videoBufferRef.getAndSet(null);
			if (vb == null) {
				return;
			}

			int[] vp = _viewport;
			gl.glViewport(vp[0], vp[1], vp[2], vp[3]);

			Size2i imageSize = vb.getImageSize();
			int w = imageSize.width;
			int h = imageSize.height;

			gl.glMatrixMode(GL.GL_PROJECTION);
			gl.glLoadIdentity();
			_glu.gluOrtho2D(0, w, h, 0);


			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, vb.getTexture());

			gl.glBegin(GL.GL_QUADS);
			gl.glTexCoord2f(0, 0); gl.glVertex2f(0, 0);
			gl.glTexCoord2f(w, 0); gl.glVertex2f(w, 0);
			gl.glTexCoord2f(w, h); gl.glVertex2f(w, h);
			gl.glTexCoord2f(0, h); gl.glVertex2f(0, h);
			gl.glEnd();
			gl.glFlush();

			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

		} finally {
			if (vb != null && !_videoBufferRef.compareAndSet(null, vb)) {
				_oldVideoBuffers.add(vb);
			}

			_displaying = false;
		}
	}

	public void reshape(GLAutoDrawable drawable, int x, int y, int width, int height) {
		// nothing to do
	}

	public void displayChanged(GLAutoDrawable drawable, boolean modeChanged, boolean deviceChanged) {
		// nothing to do
	}

	public void run() {
		_vrContext.activate();
		try {
			// TODO DebugGLの使用をやめる。でも当分はこのままにしておく。
			GLContext glContext = GLContext.getCurrent();
			glContext.setGL(new DebugGL(glContext.getGL()));

			Time duration = _mediaInput.getDuration();
			if (duration == null) {
				Time time = Time.fromFrameNumber(0, FrameDuration.FPS_29_97);
				_vrContext.reset();
				_vrContext.setRenderResolution(_resolution);
				_vrContext.setTime(time);
				_vrContext.setColorMode(ColorMode.RGBA8);

				VideoBuffer vb = null;
				try {
					vb = _mediaInput.getVideoFrameImage();
				} catch (Exception e) {
					_logger.error("error getting video frame", e);
				} catch (InternalError e) {
					if (!e.getMessage().contains("glGetError")) throw e;
					_logger.error("error getting video frame", e);
				}

				if (vb != null) {
					_vrContext.getGL().glFlush();
					_videoBufferRef.getAndSet(vb);
					_canvas.repaint();
					fireThreadRender(time);
				}
				fireThreadEndOfDuration(time);

				while (waitForRendering() != null) {
					_forceRender = false;
					fireThreadEndOfDuration(time);
				}
				return;
			}

			_forceRender = true;

			long prevFrame = -1;
			Time[] timeAndBase;
			Time prevBase = _timeKeeper.getTimeAndBase()[1];
			Time delay = TIME0;

			while ((timeAndBase = waitForRendering()) != null) {
				boolean forceRender = !_playing && _forceRender;
				_forceRender = false;

				Time frameDuration = _frameDuration;
				if (frameDuration == null) {
					frameDuration = _mediaInput.getVideoFrameDuration();
				}

				boolean seek = (timeAndBase[1] != prevBase);
				prevBase = timeAndBase[1];

				Time compensatedTime = seek ? timeAndBase[0] : timeAndBase[0].add(delay);
				delay = TIME0;

				long frameNumber = compensatedTime.toFrameNumber(frameDuration);
				boolean early = false;
				if (!seek) {
					if (forceRender) {
						frameNumber = Math.max(frameNumber, prevFrame);
					} else if (frameNumber <= prevFrame) {
						frameNumber = prevFrame + 1;
						early = true;
					}
				}

				Time frameTime = Time.fromFrameNumber(frameNumber, frameDuration);

				boolean fireEndOfDuration = false;
				if (!frameTime.before(duration)) {
					Time lastFramePlusOne = duration.add(new Time(frameDuration.timeValue-1, frameDuration.timeScale));
					frameNumber = Math.max(lastFramePlusOne.toFrameNumber(frameDuration)-1, 0);
					frameTime = Time.fromFrameNumber(frameNumber, frameDuration);
					fireEndOfDuration = _playing;
					synchronized (_monitor) {
						_playing = false;
					}
				}

				if (forceRender) {
					_timeKeeper.setTime(frameTime);
				}

				_vrContext.reset();
				_vrContext.setRenderResolution(_resolution);
				_vrContext.setTime(frameTime);
				_vrContext.setColorMode(ColorMode.RGBA8);

				VideoBuffer vb = null;

				if (_mediaItem instanceof CompositionItem) {
					// TODO _mediaInput.getDuration() など、他の _mediaInput にアクセスしている箇所も厳密にはロックする必要がある。
					PlayerLock.readLock().lock();
					try {
						// TODO prepareExpressionはプロジェクトに構造的な変更があった場合のみ行えばよい。
						Composition comp = ((CompositionItem) _mediaItem).getComposition();
						comp.prepareExpression(_vrContext.createInitialExpressionScope());

						vb = _mediaInput.getVideoFrameImage();
					} catch (Exception e) {
						_logger.error("error getting video frame", e);
					} catch (InternalError e) {
						if (!e.getMessage().contains("glGetError")) throw e;
						_logger.error("error getting video frame", e);
					} finally {
						PlayerLock.readLock().unlock();
					}
				} else {
					try {
						vb = _mediaInput.getVideoFrameImage();
					} catch (Exception e) {
						_logger.error("error getting video frame", e);
					} catch (InternalError e) {
						if (!e.getMessage().contains("glGetError")) throw e;
						_logger.error("error getting video frame", e);
					}
				}

				if (vb != null) {
					_vrContext.getGL().glFlush();

					delay = calcDelay(timeAndBase);

					if (early && !sleepUntilFrameTime(frameTime, timeAndBase[1])) {
						vb.dispose();
						vb = null;
						delay = TIME0;

					} else {
						VideoBuffer old = _videoBufferRef.getAndSet(vb);
						_canvas.repaint();
						if (old != null) {
							old.dispose();
						}

						fireThreadRender(frameTime);

						if (_logger.isTraceEnabled() && frameNumber > prevFrame + 1) {
							_logger.trace(frameNumber - prevFrame - 1 + " frames dropped");
						}

						prevFrame = frameNumber;
					}
				}

				disposeOldVideoBuffers();

				if (fireEndOfDuration) {
					fireThreadEndOfDuration(frameTime);
				}
			}

		} finally {
			while (_displaying) {
				sleep(FrameDuration.FPS_29_97);
			}

			VideoBuffer vb = _videoBufferRef.getAndSet(null);
			if (vb != null) {
				vb.dispose();
			}
			disposeOldVideoBuffers();

			_vrContext.deactivate();
		}
	}

	public void forceRender() {
		synchronized (_monitor) {
			_forceRender = true;
			_monitor.notify();
		}
	}

	public void play(boolean play) {
		if (_playing == play) {
			return;
		}

		synchronized (_monitor) {
			_playing = play;
			_monitor.notify();
		}
	}

	public void finish() {
		if (_finished) {
			return;
		}

		synchronized (_monitor) {
			_finished = true;
			_monitor.notify();
		}

		try {
			join();
		} catch (InterruptedException e) {
			// ignore
		}

		cleanup();
	}

	private Time[] waitForRendering() {
		if (!_playing && !_forceRender) {
			synchronized (_monitor) {
				while (!_playing && !_forceRender && !_finished) {
					try {
						_monitor.wait();
					} catch (InterruptedException e) {
						// ignore
					}
				}
			}
		}
		return _finished ? null : _timeKeeper.getTimeAndBase();
	}

	private void sleep(Time time) {
		if (time.timeValue > 0) {
			try {
				long sleep = (long)(time.toSecond() * 1000000000);
				Thread.sleep(sleep / 1000000, (int) (sleep % 1000000));
			} catch (InterruptedException e) {
			}
		}
	}

	private Time calcDelay(Time[] timeAndBase) {
		Time[] curTimeAndBase = _timeKeeper.getTimeAndBase();
		if (curTimeAndBase[1] != timeAndBase[1]) {
			// シークした場合。
			return TIME0;
		}

		Time delay = curTimeAndBase[0].subtract(timeAndBase[0]);
		return (delay.timeValue > 0) ? delay : TIME0;
	}

	private boolean sleepUntilFrameTime(Time frameTime, Time tkBase) {
		while (_playing && !_finished) {
			Time[] curTimeAndBase = _timeKeeper.getTimeAndBase();
			if (curTimeAndBase[1] != tkBase) {
				// シークした場合。
				return false;
			}

			Time sleepTime = frameTime.subtract(curTimeAndBase[0]);
			if (sleepTime.timeValue <= 0) {
				return true;
			}

			sleep(sleepTime);
		}

		return false;
	}

	private void disposeOldVideoBuffers() {
		for (Iterator<VideoBuffer> it = _oldVideoBuffers.iterator(); it.hasNext(); ) {
			VideoBuffer old = it.next();
			old.dispose();
			it.remove();
		}
	}

	public void addPlayerThreadListener(PlayerThreadListener listener) {
		_playerThreadListeners.add(listener);
	}

	public void removePlayerThreadListener(PlayerThreadListener listener) {
		_playerThreadListeners.remove(listener);
	}

	private void fireThreadRender(Time time) {
		PlayerThreadEvent event = new PlayerThreadEvent(this, time);
		for (Object l : _playerThreadListeners.getListeners()) {
			((PlayerThreadListener) l).threadRender(event);
		}
	}

	private void fireThreadEndOfDuration(Time time) {
		PlayerThreadEvent event = new PlayerThreadEvent(this, time);
		for (Object l : _playerThreadListeners.getListeners()) {
			((PlayerThreadListener) l).threadEndOfDuration(event);
		}
	}

}
