/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collection;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class ClearEffectKeyframesOperation<V> extends ProjectOperation {

	private final String _compId;

	private final String _layerId;

	private final int _effectIndex;

	private final String _property;

	private final String _data;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public ClearEffectKeyframesOperation(
			ProjectManager projectManager, EffectableLayer layer, int effectIndex, String property) {

		super(projectManager, "ストップウォッチの解除");
		InjectorHolder.getInjector().injectMembers(this);

		LayerComposition comp = projectManager.checkLayer(layer);
		Effect effect = layer.getEffects().get(effectIndex);
		AnimatableValue<V> avalue = PropertyUtil.getProperty(effect, property);

		if (!avalue.hasKeyframe()) {
			throw new IllegalArgumentException("no keyframes");
		}

		_compId = comp.getId();
		_layerId = layer.getId();
		_effectIndex = effectIndex;
		_property = property;
		_data = _encoder.encodeElement(avalue);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		clearOrRestore(true, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		clearOrRestore(false, project, pm);
		return Status.OK_STATUS;
	}

	private void clearOrRestore(boolean clear, Project project, ProjectManager pm) throws ExecutionException {
		LayerComposition comp = project.getComposition(_compId);
		EffectableLayer layer = (EffectableLayer) comp.getLayer(_layerId);
		Effect effect = layer.getEffects().get(_effectIndex);
		AnimatableValue<V> avalue = PropertyUtil.getProperty(effect, _property);

		if (clear) {
			avalue.clearKeyframes();
			if (pm != null) {
				pm.fireKeyframesChange(comp, new Object[][] { new Object[] { layer, _effectIndex, _property, null } });
			}

		} else {
			restore(avalue);
			if (pm != null) {
				Collection<Keyframe<V>> keyframes = avalue.getKeyframes();
				Object[][] data = new Object[keyframes.size()][];
				int i = 0;
				for (Keyframe<V> kf : keyframes) {
					data[i++] = new Object[] { layer, _effectIndex, _property, kf.time };
				}
				pm.fireKeyframesChange(comp, data);
			}
		}
	}

	// TODO ThreeDSwitchOperation 等にほぼ同じメソッドがあるので整理する。
	private <A extends AnimatableValue<V>> void restore(A dst) throws ExecutionException {
		try {
			@SuppressWarnings("unchecked")
			A src = _decoder.decodeElement(_data, (Class<A>) dst.getClass());

			dst.clearKeyframes(src.getStaticValue());
			dst.setExpression(src.getExpression());

			for (Keyframe<V> kf : src.getKeyframes()) {
				dst.putKeyframe(kf);
			}

		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding AnimatableValue data", e);
		}
	}

}
