/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Map;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class ShiftKeyframesOperation extends ProjectOperation {

	private final String _compId;

	private final Object[][] _keyframeData;

	private final String[] _savedAvalues;

	private String[] _newAvalues;

	private Time _shiftTime;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public ShiftKeyframesOperation(
			ProjectManager projectManager, LayerComposition comp,
			Object[][] keyframeData, String[] baseAnimatableValues,
			Time shiftTime, String relation) {

		super(projectManager, "キーフレームの移動", relation);
		InjectorHolder.getInjector().injectMembers(this);

		projectManager.checkComposition(comp);
		_compId = comp.getId();
		_keyframeData = keyframeData;
		_savedAvalues = new String[_keyframeData.length];
		_newAvalues = new String[_keyframeData.length];
		_shiftTime = shiftTime;

		init(comp, baseAnimatableValues);
	}

	private void init(LayerComposition comp, String[] baseAnimatableValues) {
		AnimatableValue<?>[] newAvalues = new AnimatableValue[_keyframeData.length];
		Map<AnimatableValue<?>, AnimatableValue<?>> map = Util.newMap();

		for (int i = 0, n = _keyframeData.length; i < n; ++i) {
			Layer layer = comp.getLayer((String) _keyframeData[i][0]);
			Integer effectIndex = (Integer) _keyframeData[i][1];
			String property = (String) _keyframeData[i][2];
			Keyframe<?> kf = (Keyframe<?>) _keyframeData[i][3];

			AnimatableValue<?> avalue;
			if (effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				avalue = PropertyUtil.getProperty(effect, property);
			}

			AnimatableValue<?> newAvalue = map.get(avalue);
			if (newAvalue == null) {
				_savedAvalues[i] = _encoder.encodeElement(avalue);
				try {
					newAvalue = _decoder.decodeElement(baseAnimatableValues[i], avalue.getClass());
				} catch (ProjectDecodeException e) {
					throw new JavieRuntimeException(e);
				}
				map.put(avalue, newAvalue);
				newAvalues[i] = newAvalue;
			}

			@SuppressWarnings("unchecked")
			AnimatableValue<Object> tmp = (AnimatableValue<Object>) newAvalue;
			tmp.putKeyframe(kf.time.add(_shiftTime), kf.value, kf.interpolation);
		}

		for (int i = 0, n = newAvalues.length; i < n; ++i) {
			if (newAvalues[i] != null) {
				_newAvalues[i] = _encoder.encodeElement(newAvalues[i]);
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		shiftOrRestore(true, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		shiftOrRestore(false, project, pm);
		return Status.OK_STATUS;
	}

	private void shiftOrRestore(boolean shift, Project project, ProjectManager pm) throws ExecutionException {
		LayerComposition comp = project.getComposition(_compId);
		Object[][] data = (pm != null) ? new Object[_keyframeData.length][] : null;

		for (int i = 0, n = _keyframeData.length; i < n; ++i) {
			Layer layer = comp.getLayer((String) _keyframeData[i][0]);
			Integer effectIndex = (Integer) _keyframeData[i][1];
			String property = (String) _keyframeData[i][2];
			Keyframe<?> kf = (Keyframe<?>) _keyframeData[i][3];

			AnimatableValue<?> avalue;
			if (effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				avalue = PropertyUtil.getProperty(effect, property);
			}

			if (shift) {
				if (_newAvalues[i] != null) {
					restore(_newAvalues[i], avalue);
				}
				if (pm != null) {
					data[i] = new Object[] { layer, effectIndex, property, kf.time.add(_shiftTime) };
				}
			} else {
				if (_savedAvalues[i] != null) {
					restore(_savedAvalues[i], avalue);
				}
				if (pm != null) {
					data[i] = new Object[] { layer, effectIndex, property, kf.time };
				}
			}
		}

		if (pm != null) {
			pm.fireKeyframesChange(comp, data);
		}
	}

	// TODO ThreeDSwitchOperation 等にほぼ同じメソッドがあるので整理する。
	private <A extends AnimatableValue<V>, V> void restore(String data, A dst) throws ExecutionException {
		try {
			@SuppressWarnings("unchecked")
			A src = _decoder.decodeElement(data, (Class<A>) dst.getClass());

			dst.clearKeyframes(src.getStaticValue());
			dst.setExpression(src.getExpression());

			for (Keyframe<V> kf : src.getKeyframes()) {
				dst.putKeyframe(kf);
			}

		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding AnimatableValue data", e);
		}
	}

	@Override
	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {

		if (pm == null) {
			// _newAvalues, _shiftTime の値は、シャドウプロジェクトに反映される前に
			// 後続のマージ可能オペレーションで上書きされているかもしれない。
			// しかし、シャドウオペレーションのキューが全て実行された時点での値は一致するので問題ない。
			try {
				synchronized (this) {
					shiftOrRestore(true, project, null);
				}
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			}
			return true;
		}

		if (!(operation instanceof ShiftKeyframesOperation)) {
			return false;
		}

		ShiftKeyframesOperation newOp = (ShiftKeyframesOperation) operation;
		if (!newOp._compId.equals(_compId) || newOp._keyframeData != _keyframeData) {
			return false;
		}

		if (!newOp._shiftTime.equals(_shiftTime)) {
			synchronized (this) {
				_newAvalues = newOp._newAvalues;
				_shiftTime = newOp._shiftTime;
			}
			try {
				shiftOrRestore(true, project, pm);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			}

			// 結合の結果、このオペレーションの直前の値に戻ったかどうか。
			_noEffect = (_shiftTime.timeValue == 0);
		}

		return true;
	}

}
