/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.ViewPart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.app.UIUtil;
import ch.kuramo.javie.app.player.MediaPlayer;
import ch.kuramo.javie.app.project.ProjectEvent;
import ch.kuramo.javie.app.project.ProjectEventHub;
import ch.kuramo.javie.app.project.ProjectListener;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.MediaItem;

public class MediaPlayerView extends ViewPart implements ProjectListener, IPartListener {

	public static final String ID = "ch.kuramo.javie.app.views.mediaPlayerView";


	private static final Logger _logger = LoggerFactory.getLogger(MediaPlayerView.class);


	private MediaItem _mediaItem;

	private MediaPlayer _mediaPlayer;


	public void createPartControl(Composite parent) {
		ProjectManager pm = ProjectManager.forWorkbenchWindow(getSite().getWorkbenchWindow());
		if (pm == null) {
			setPartName("");
		} else if (!setProjectManager(pm)) {
			throw new IllegalStateException("no such MediaItem found: " + getItemId());
		}

		_mediaPlayer = new MediaPlayer(parent, SWT.NULL);

		if (_mediaItem != null) {
			_mediaPlayer.setMediaItem(_mediaItem);

			if (_mediaItem instanceof CompositionItem) {
				LayerCompositionView.connectToPlayer(this);
			}
		}

		getSite().getPage().addPartListener(this);
		ProjectEventHub.getHub().addProjectListener(this);
	}

	public void dispose() {
		ProjectEventHub.getHub().removeProjectListener(this);
		getSite().getPage().removePartListener(this);
		super.dispose();
	}

	public void setFocus() {
		_mediaPlayer.setFocus();
	}

	public void handleProjectEvent(ProjectEvent event) {
		ProjectManager pm = event.getProjectManager();
		if (pm.getWorkbenchWindow() != getSite().getWorkbenchWindow()) {
			return;
		}

		switch (event.type) {
			case PROJECT_INITIALIZE:
				if (setProjectManager(pm)) {
					_mediaPlayer.setMediaItem(_mediaItem);

					if (_mediaItem instanceof CompositionItem) {
						LayerCompositionView.connectToPlayer(this);
					}
				} else {
					_logger.warn("no such MediaItem found: " + getItemId());
					UIUtil.hideView(getSite().getPage(), this);
				}
				break;

			case PROJECT_DISPOSE:
				UIUtil.hideView(getSite().getPage(), this);
				break;
		}
	}

	String getItemId() {
		return getViewSite().getSecondaryId();
	}

	MediaPlayer getMediaPlayer() {
		return _mediaPlayer;
	}

	private boolean setProjectManager(ProjectManager pm) {
		_mediaItem = pm.getShadow().getItem(getItemId());
		if (_mediaItem == null) {
			return false;
		}

		setPartName(_mediaItem.getName());
		return true;
	}

	public void partClosed(IWorkbenchPart part) {
		if (part == this) {
			_mediaPlayer.close();
		}
	}

	public void partOpened(IWorkbenchPart part) {
		// nothing to do
	}

	public void partActivated(IWorkbenchPart part) {
		// nothing to do
	}

	public void partDeactivated(IWorkbenchPart part) {
		// nothing to do
	}

	public void partBroughtToTop(IWorkbenchPart part) {
		// nothing to do
	}

}
