/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Event;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;

import ch.kuramo.javie.app.ImageUtil;
import ch.kuramo.javie.app.project.ModifyLayerParentOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.ComboBoxViewerCellEditor;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.ItemLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.NullLayer;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.TransformableLayer;
import ch.kuramo.javie.core.Util;

public class LayerElement extends Element {

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");


	public final Layer layer;

	private final List<Element> children = Util.newList();

	private final Map<Class<? extends Element>, Element> childrenMap = Util.newMap();

	private final SwitchGroup showHideColumnSwitches = new SwitchGroup();

	private final SwitchGroup switchesColumnSwitches = new SwitchGroup();

	private ComboBoxViewerCellEditor editor;


	public LayerElement(TreeViewer viewer, Layer layer) {
		super(viewer);
		this.layer = layer;

		ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);

		if (LayerNature.isVideoNature(layer)) {
			showHideColumnSwitches.add(new VideoSwitch(pm, layer));
		} else {
			showHideColumnSwitches.addPlaceholder();
		}

		if (LayerNature.isAudioNature(layer)) {
			showHideColumnSwitches.add(new AudioSwitch(pm, layer));
		} else {
			showHideColumnSwitches.addPlaceholder();
		}


		switchesColumnSwitches.add(new ShySwitch(pm, layer));

		if (layer instanceof EffectableLayer) {
			switchesColumnSwitches.add(new EffectsSwitch(pm, (EffectableLayer) layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}

		if (LayerNature.isCTCRNature(layer)) {
			switchesColumnSwitches.add(new CTCRSwitch(pm, layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}

		if (layer instanceof TransformableLayer && LayerNature.isThreeDNature(layer)) {
			switchesColumnSwitches.add(new ThreeDSwitch(pm, (TransformableLayer) layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}
	}

	public Image getColumnImage(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				if (layer instanceof ItemLayer) {
					return ImageUtil.getItemIcon(((ItemLayer) layer).getItem());
				}
				if (layer instanceof CameraLayer) {
					return ImageUtil.getCameraIcon();
				}
				if (layer instanceof NullLayer) {
					return ImageUtil.getNullIcon();
				}
				if (layer instanceof TextLayer) {
					return ImageUtil.getTextIcon();
				}

				// TODO その他のレイヤーのアイコン
				return PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJ_ELEMENT);

			default:
				return null;
		}
	}

	public String getColumnText(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL: {
				String name = layer.getName();
				if (layer instanceof ItemLayer) {
					ItemLayer itemLayer = (ItemLayer) layer;
					String itemName = itemLayer.getItem().getName();
					if (!name.equals(itemName)) {
						name = String.format("%s  <%s>", name, itemName);
					}
				}
				return name;
			}

			case LayerCompositionView.PARENT_COL:
				Layer parent = layer.getParent();
				return (COCOA ? "  " : "") + ((parent != null) ? parent.getName() : "[ なし ]");

			default:
				return null;
		}
	}

	private <E extends Element> E getChildElement(Class<E> clazz) {
		@SuppressWarnings("unchecked")
		E element = (E) childrenMap.get(clazz);
		if (element == null) {
			try {
				element = clazz.getConstructor(getClass()).newInstance(this);
			} catch (Exception e) {
				throw new JavieRuntimeException(e);
			}
			childrenMap.put(clazz, element);
		}
		return element;
	}

	private void prepareChildren() {
		children.clear();

		if (LayerNature.isTimeRemapEnabled(layer)) {
			children.add(getChildElement(TimeRemapElement.class));
		}

		if (layer instanceof TextLayer) {
			children.add(getChildElement(TextElement.class));
		}

		if (layer instanceof EffectableLayer && !((EffectableLayer) layer).getEffects().isEmpty()) {
			children.add(getChildElement(EffectsElement.class));
		}

		if (layer instanceof TransformableLayer && LayerNature.isVideoNature(layer)) {
			children.add(getChildElement(TransformElement.class));

		} else if (layer instanceof CameraLayer) {
			children.add(getChildElement(CameraTransformElement.class));
			children.add(getChildElement(CameraOptionsElement.class));
		}

		if (layer instanceof MediaLayer && LayerNature.isAudioNature(layer)) {
			children.add(getChildElement(AudioElement.class));
		}

		childrenMap.values().retainAll(children);
	}

	@Override
	public boolean hasChildren() {
		prepareChildren();
		return (children.size() > 0);
	}

	@Override
	public Element[] getChildren() {
		// getChildren が呼ばれるより前に必ず hasChildren が呼ばれるようではあるが、
		// 念のため prepareChildren を実行しておく。
		prepareChildren();
		return children.toArray(new Element[children.size()]);
	}

	public void paintColumn(Event event) {
		switch (event.index) {
			case LayerCompositionView.SHOWHIDE_COL:
				showHideColumnSwitches.paint(event);
				break;

			case LayerCompositionView.VALUE_COL:
				switchesColumnSwitches.paint(event);
				break;

			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				tm.drawLayer(event, layer);
				break;

			default:
				super.paintColumn(event);
				break;
		}
	}

	public void updateCursor(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				tm.updateCursor(event, layer);
				break;

			default:
				super.updateCursor(event, columnIndex);
				break;
		}
	}

	public void mouseDown(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.SHOWHIDE_COL:
				showHideColumnSwitches.mouseDown(event);
				break;

			case LayerCompositionView.VALUE_COL:
				switchesColumnSwitches.mouseDown(event);
				break;

			case LayerCompositionView.PARENT_COL:
				// WIN32では選択中の行でしかエディタが起動しないので、プログラム上からエディタを起動する。
				// (おそらくFULL_SELECTIONにすれば選択中でなくても(行のどこでもクリックすれば行が選択されるので、結果的に)エディタは起動するはずだが)
				// COCOAではプログラム上からエディタを起動しないと挙動がおかしくなる。
				if (event.button == 1) {
					viewer.editElement(this, LayerCompositionView.PARENT_COL);
				}
				break;

			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				tm.mouseDown(event, layer);
				break;

			default:
				super.mouseDown(event, columnIndex);
				break;
		}
	}

	public boolean canEdit(int columnIndex) {
		if (columnIndex == LayerCompositionView.PARENT_COL) {
			return true;
		}
		return false;
	}

	public CellEditor getCellEditor(int columnIndex) {
		if (columnIndex == LayerCompositionView.PARENT_COL) {
			if (editor == null) {
				editor = new ComboBoxViewerCellEditor(viewer.getTree(), SWT.READ_ONLY);
				editor.setContenProvider(ArrayContentProvider.getInstance());
				editor.setLabelProvider(new ParentLabelProvider());
				editor.setActivationStyle(ComboBoxViewerCellEditor.DROP_DOWN_ON_PROGRAMMATIC_ACTIVATION);

				((Combo) editor.getControl()).addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent e) {
						setCellEditorValue(LayerCompositionView.PARENT_COL, editor.getValue());
					}
				});
			}

			Object[] layers = getLayersForParentEditor();
			editor.setInput(layers);

			return editor;
		}
		return null;
	}

	public Object getCellEditorValue(int columnIndex) {
		if (columnIndex == LayerCompositionView.PARENT_COL) {
			Layer parent = layer.getParent();
			return (parent != null) ? parent : NO_PARENT;
		}
		return null;
	}

	public void setCellEditorValue(int columnIndex, Object value) {
		if (columnIndex == LayerCompositionView.PARENT_COL) {
			ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
			pm.postOperation(new ModifyLayerParentOperation(pm, layer, (value != NO_PARENT) ? (Layer) value : null));
		}
	}

	private Object[] getLayersForParentEditor() {
		List<Object> layers = Util.newList();

		LayerComposition comp = (LayerComposition) viewer.getData(LayerCompositionView.LAYER_COMPOSITION);
		for (Layer l : comp.getLayers()) {
			Layer tmp = l;
			while (tmp != null && tmp != layer) {
				tmp = tmp.getParent();
			}
			if (tmp == null) {
				layers.add(l);
			}
		}

		layers.add(NO_PARENT);
		Collections.reverse(layers);

		return layers.toArray();
	}

	private static final Object NO_PARENT = new Object();

	private static class ParentLabelProvider extends LabelProvider {

		public String getText(Object element) {
			return (element != NO_PARENT) ? ((Layer) element).getName() : "[ なし ]";
		}

	}

}
