/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.wizards;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.widgets.FrameRateComboUtil;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.core.FrameDuration;

public class CompositionWizardPage extends WizardPage {

	// TODO この最大値は暫定。グラフィックカード毎に異なるはず。
	private static final int MAX_SIZE = 8192;

	private CompositionSettings _settings;

	private Text _nameText;

	private Text _widthText;

	private Text _heightText;

	private Button _freezeRatio;

	private Combo _fpsCombo;

	private Text _durationText;

	private Combo _colorModeCombo;

	private ModifyListener _modifyListener;

	private int _widthRatio;

	private int _heightRatio;


	public CompositionWizardPage(CompositionSettings settings) {
		super("CompositionWizardPage", "コンポジション", null);
		setDescription("コンポジションを作成します。");
		_settings = settings;
	}

	public void createControl(Composite parent) {
		String[] colorModeItems = {
				"8 bit/チャンネル",
				"16 bit/チャンネル",
				"16 bit/チャンネル（浮動小数点）",
				"32 bit/チャンネル（浮動小数点）"		
		};

		GridBuilder gb = new GridBuilder(parent, 16, true);

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "名前:");
		_nameText =			gb.hSpan(12).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, _settings.name);

							gb.hSpan(16).size(10, 10).composite(SWT.NULL);

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "幅:");
		_widthText =		gb.hSpan(3).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, String.valueOf(_settings.width));
		_freezeRatio =		gb.span(9, 2).hAlign(SWT.FILL).hGrab().button(SWT.CHECK, "縦横比を固定");

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "高さ:");
		_heightText =		gb.hSpan(3).hAlign(SWT.FILL).hGrab().tabAfter(_widthText)
								.text(SWT.BORDER, String.valueOf(_settings.height));

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "フレームレート:");
		_fpsCombo = 		gb.hSpan(3).hAlign(SWT.FILL).hGrab().combo(SWT.NULL, 0,
								FrameRateComboUtil.getComboItems(), FrameRateComboUtil.toComboItem(_settings.frameDuration));
							gb.hSpan(9).hGrab().label(SWT.NULL, "フレーム/秒");

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "デュレーション:");
		_durationText =		gb.hSpan(3).hAlign(SWT.FILL).hGrab()
								.text(SWT.BORDER, String.valueOf(_settings.duration.toFrameNumber(_settings.frameDuration)));
							gb.hSpan(9).hGrab().label(SWT.NULL, "フレーム");

							gb.hSpan(16).size(10, 10).composite(SWT.NULL);

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "色深度:");
		_colorModeCombo =	gb.hSpan(12).hGrab().combo(SWT.READ_ONLY, colorModeItems, _settings.colorMode.ordinal());


		_modifyListener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				doValidate((Control) e.getSource());
			}
		};

		SelectionListener selectionListener = new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				doValidate((Control) e.getSource());
			}
			public void widgetDefaultSelected(SelectionEvent e) {
				doValidate((Control) e.getSource());
			}
		};

		_nameText.addModifyListener(_modifyListener);

		_widthText.addModifyListener(_modifyListener);
		_heightText.addModifyListener(_modifyListener);

		_freezeRatio.addSelectionListener(selectionListener);

		_fpsCombo.addModifyListener(_modifyListener);
		_fpsCombo.addSelectionListener(selectionListener);

		_durationText.addModifyListener(_modifyListener);


		Composite grid = gb.getComposite();
		grid.setTabList(gb.getTabList());

		doValidate(null);
		setControl(grid);
	}

	private void doValidate(Control control) {
		setPageComplete(false);

		String wStr = _widthText.getText().trim();
		int width;
		try {
			width = Integer.parseInt(wStr);
		} catch (NumberFormatException e) {
			width = 0;
		}

		String hStr = _heightText.getText().trim();
		int height;
		try {
			height = Integer.parseInt(hStr);
		} catch (NumberFormatException e) {
			height = 0;
		}

		boolean frozen = _freezeRatio.getSelection();
		if (frozen) {
			if (control == _widthText && width > 0) {
				height = Math.max(1, width * _heightRatio / _widthRatio);
				_heightText.removeModifyListener(_modifyListener);
				_heightText.setText(String.valueOf(height));
				_heightText.addModifyListener(_modifyListener);
			} else if (control == _heightText && height > 0) {
				width = Math.max(1, height * _widthRatio / _heightRatio);
				_widthText.removeModifyListener(_modifyListener);
				_widthText.setText(String.valueOf(width));
				_widthText.addModifyListener(_modifyListener);
			}
		} else if ((control == null || control == _widthText || control == _heightText) && width > 0 && height > 0) {
			int gcd = gcd(width, height);
			_widthRatio = width / gcd;
			_heightRatio = height / gcd;
			_freezeRatio.setText(String.format("縦横比を%d:%dに固定", _widthRatio, _heightRatio));
		}

		if (_nameText.getText().trim().length() == 0) {
			setErrorMessage("名前が入力されていません。");
			return;
		}
		if (wStr.length() == 0) {
			setErrorMessage("幅が入力されていません。");
			return;
		}
		if (width < 1 || width > MAX_SIZE) {
			setErrorMessage(String.format("幅には1から%dまでの整数を入力してください。", MAX_SIZE));
			return;
		}
		if (hStr.length() == 0) {
			setErrorMessage("高さが入力されていません。");
			return;
		}
		if (height < 1 || height > MAX_SIZE) {
			setErrorMessage(String.format("高さには1から%dまでの整数を入力してください。", MAX_SIZE));
			return;
		}

		String fpsStr = _fpsCombo.getText().trim();
		if (fpsStr.length() == 0) {
			setErrorMessage("フレームレートが入力されていません。");
			return;
		}
		double fps = 0;
		try {
			fps = Double.parseDouble(fpsStr);
		} catch (NumberFormatException e) {
		}
		if (fps <= 0) {
			setErrorMessage("フレームレートには正の数値を入力してください。");
			return;
		}

		String durationStr = _durationText.getText().trim();
		if (durationStr.length() == 0) {
			setErrorMessage("デュレーションが入力されていません。");
			return;
		}
		int duration = 0;
		try {
			duration = Integer.parseInt(durationStr);
		} catch (NumberFormatException e) {
		}
		if (duration <= 0) {
			setErrorMessage("デュレーションには1以上の整数を入力してください。");
			return;
		}


		setErrorMessage(null);
		setPageComplete(true);
	}

	private int gcd(int x, int y) {
		while (y != 0) {
			int t = x % y;
			x = y;
			y = t;
		}
		return x;
	}

	public CompositionSettings getResult() {
		if (!isPageComplete()) {
			return null;
		}

		CompositionSettings result = new CompositionSettings();
		result.name = _nameText.getText().trim();
		result.width = Integer.parseInt(_widthText.getText().trim());
		result.height = Integer.parseInt(_heightText.getText().trim());
		result.frameDuration = FrameRateComboUtil.toFrameDuration(_fpsCombo.getText().trim());
		result.duration = Time.fromFrameNumber(Integer.valueOf(_durationText.getText().trim()), result.frameDuration);
		result.colorMode = ColorMode.values()[_colorModeCombo.getSelectionIndex()];
		return result;
	}

}

class CompositionSettings implements Cloneable {

	String name = "コンポ";

	int width = 640;

	int height = 480;

	Time frameDuration = FrameDuration.FPS_29_97;

	Time duration = Time.fromFrameNumber(4000, FrameDuration.FPS_29_97);

	ColorMode colorMode = ColorMode.RGBA8;


	CompositionSettings() {
	}

	CompositionSettings(String name, int width, int height,
			Time frameDuration, Time duration, ColorMode colorMode) {
		this.name = name;
		this.width = width;
		this.height = height;
		this.frameDuration = frameDuration;
		this.duration = duration;
		this.colorMode = colorMode;
	}

	public CompositionSettings clone() {
		try {
			return (CompositionSettings) super.clone();
		} catch (CloneNotSupportedException e) {
			throw new Error(e);
		}
	}

}
