/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;

import javax.vecmath.AxisAngle4d;
import javax.vecmath.Matrix4d;
import javax.vecmath.Quat4d;

import ch.kuramo.javie.api.Time;

public abstract class AbstractAnimatableValue<V> implements AnimatableValue<V> {

	private V _staticValue;

	private final SortedMap<Time, Keyframe<V>> _keyframes;

	private String _expression;


	public AbstractAnimatableValue(V staticValue, Collection<Keyframe<V>> keyframes, String expression) {
		super();
		_staticValue = staticValue;
		_keyframes = Util.newSortedMap();
		_expression = expression;

		if (keyframes != null) {
			for (Keyframe<V> k : keyframes) {
				putKeyframe(k);
			}
		}
	}

	public AbstractAnimatableValue(V staticValue) {
		this(staticValue, null, null);
	}

	public V value(RenderContext renderContext) {
		if (_expression != null && !renderContext.checkCyclicEvaluation(this)) {
			try {
				V value = renderContext.evaluateExpression(this);
				if (value != null) {
					return value;
				}
				throw new NullPointerException();
			} catch (Exception e) {
				// TODO エラーが発生しているかどうかを示すフラグを立てる。_expressionを変更した後はそのフラグはリセットする。
				// TODO そのフラグを立てたとき、つまり初回だけエクスプレッションエラーをGUIに通知する仕組みを作る
				e.printStackTrace();
			}
		}
		return value(renderContext.getTime());
	}

	public V valueAtTime(final Time time, final RenderContext renderContext) {
		if (_expression != null && !renderContext.checkCyclicEvaluation(this)) {
			try {
				V value = renderContext.saveAndExecute(new WrappedOperation<V>() {
					public V execute() {
						renderContext.setTime(time);
						return renderContext.evaluateExpression(AbstractAnimatableValue.this);
					}
				});
				if (value != null) {
					return value;
				}
				throw new NullPointerException();
			} catch (Exception e) {
				// TODO エラーが発生しているかどうかを示すフラグを立てる。_expressionを変更した後はそのフラグはリセットする。
				// TODO そのフラグを立てたとき、つまり初回だけエクスプレッションエラーをGUIに通知する仕組みを作る
				e.printStackTrace();
			}
		}
		return value(time);
	}

	private V value(Time time) {
		if (_keyframes.isEmpty()) {
			return _staticValue;
		}

		Keyframe<V> k1 = null;
		Keyframe<V> k2 = null;

		Iterator<Map.Entry<Time, Keyframe<V>>> tail = _keyframes.tailMap(time).entrySet().iterator();
		if (tail.hasNext()) {
			Map.Entry<Time, Keyframe<V>> entry = tail.next();
			if (entry.getKey().equals(time)) {
				k1 = entry.getValue();
				if (tail.hasNext()) {
					k2 = tail.next().getValue();
				}
			} else {
				k2 = entry.getValue();
			}
		}
		if (k1 == null) {
			SortedMap<Time, Keyframe<V>> headMap = _keyframes.headMap(time);
			if (!headMap.isEmpty()) {
				k1 = headMap.get(headMap.lastKey());
			}
		}

		if (k1 == null) {
			return k2.value;
		} if (k2 == null) {
			return k1.value;
		}

		switch (k1.interpolation) {
			case HOLD:
				return k1.value;
			case LINEAR:
				return valueOf(linear(toArray(k1.value), toArray(k2.value), k1.time, k2.time, time));

			case CATMULL_ROM: {
				SortedMap<Time, Keyframe<V>> headMap = _keyframes.headMap(k1.time);
				Keyframe<V> k0 = !headMap.isEmpty() ? headMap.get(headMap.lastKey()) : k1;
				Keyframe<V> k3 = tail.hasNext() ? tail.next().getValue() : k2;
				return valueOf(catmullRom(toArray(k0.value), toArray(k1.value),
						toArray(k2.value), toArray(k3.value), k1.time, k2.time, time));
			}

			case SLERP:
				return valueOf(slerp(toArray(k1.value), toArray(k2.value), k1.time, k2.time, time));

			default:
				throw new UnsupportedOperationException(k1.interpolation.name());
		}
	}

	protected abstract double[] toArray(V value);

	protected abstract V valueOf(double[] d);

	protected double[] linear(double[] p1, double[] p2, Time time1, Time time2, Time time) {
		double[] result = new double[p1.length];
		for (int i = 0, n = result.length; i < n; ++i) {
			double s1 = time1.toSecond();
			double s2 = time2.toSecond();
			double s = time.toSecond();

			double t = (s - s1) / (s2 - s1);

			result[i] = (p2[i] - p1[i]) * t + p1[i];
		}
		return result;
	}

	protected double[] catmullRom(
			double[] p0, double[] p1, double[] p2, double[] p3, Time time1, Time time2, Time time) {

		double[] result = new double[p0.length];
		for (int i = 0, n = result.length; i < n; ++i) {
			double s1 = time1.toSecond();
			double s2 = time2.toSecond();
			double s = time.toSecond();

			double t = (s - s1) / (s2 - s1);
			double tt = t*t;
			double ttt = tt*t;

			double v0 = (p2[i] - p0[i]) * 0.5;
			double v1 = (p3[i] - p1[i]) * 0.5;
			result[i] = (2*p1[i] - 2*p2[i] + v0 + v1)*ttt + (-3*p1[i] + 3*p2[i] - 2*v0 - v1)*tt + v0*t + p1[i];
		}
		return result;
	}

	protected double[] slerp(double[] p1, double[] p2, Time time1, Time time2, Time time) {
		double s1 = time1.toSecond();
		double s2 = time2.toSecond();
		double s = time.toSecond();
		double t = (s - s1) / (s2 - s1);

		Matrix4d m1 = new Matrix4d();
		Matrix4d m2 = new Matrix4d();
		Matrix4d m = new Matrix4d();
		AxisAngle4d aa = new AxisAngle4d();

		m1.setIdentity();
		aa.set(1, 0, 0, Math.toRadians(p1[0]));
		m.set(aa);
		m1.mul(m);
		aa.set(0, 1, 0, Math.toRadians(p1[1]));
		m.set(aa);
		m1.mul(m);
		aa.set(0, 0, 1, Math.toRadians(p1[2]));
		m.set(aa);
		m1.mul(m);

		m2.setIdentity();
		aa.set(1, 0, 0, Math.toRadians(p2[0]));
		m.set(aa);
		m2.mul(m);
		aa.set(0, 1, 0, Math.toRadians(p2[1]));
		m.set(aa);
		m2.mul(m);
		aa.set(0, 0, 1, Math.toRadians(p2[2]));
		m.set(aa);
		m2.mul(m);

		Quat4d q1 = new Quat4d();
		Quat4d q2 = new Quat4d();
		q1.set(m1);
		q2.set(m2);

		q1.interpolate(q2, t);

		m.set(q1);

		double x = Math.toDegrees(Math.atan2(m.m12, m.m22));
		double y = Math.toDegrees(Math.asin(-m.m02));
		double z = Math.toDegrees(Math.atan2(m.m01, m.m00));

		// TODO なぜか正負を逆にしないとダメみたい？
		return new double[] { -x, -y, -z };
	}

	public V getStaticValue() {
		return _staticValue;
	}

	public boolean hasKeyframe() {
		return !_keyframes.isEmpty();
	}

	public Keyframe<V> getKeyframe(Time time) {
		return _keyframes.get(time);
	}

	public Collection<Keyframe<V>> getKeyframes() {
		return Collections.unmodifiableCollection(_keyframes.values());
	}

	public Collection<Keyframe<V>> getKeyframes(Time fromTime, Time toTime) {
		return Collections.unmodifiableCollection(_keyframes.subMap(fromTime, toTime).values());
	}

	public SortedMap<Time, Keyframe<V>> headKeyframeMap(Time toTime) {
		return Collections.unmodifiableSortedMap(_keyframes.headMap(toTime));
	}

	public SortedMap<Time, Keyframe<V>> tailKeyframeMap(Time fromTime) {
		return Collections.unmodifiableSortedMap(_keyframes.tailMap(fromTime));
	}

	public abstract Set<Interpolation> supportedInterpolations();

	public Keyframe<V> putKeyframe(Keyframe<V> keyframe) {
		if (!supportedInterpolations().contains(keyframe.interpolation)) {
			throw new IllegalArgumentException("unsupported interpolation: " + keyframe.interpolation.name());
		}
		return _keyframes.put(keyframe.time, keyframe);
	}

	public Keyframe<V> putKeyframe(Time time, V value, Interpolation interpolation) {
		return putKeyframe(new Keyframe<V>(time, value, interpolation));
	}

	public Keyframe<V> removeKeyframe(Time time) {
		Keyframe<V> removed = _keyframes.remove(time);
		if (_keyframes.isEmpty() && removed != null) {
			_staticValue = removed.value;
		}
		return removed;
	}

	public void clearKeyframes() {
		if (!_keyframes.isEmpty()) {
			clearKeyframes(_keyframes.get(_keyframes.firstKey()).value);
		}
	}

	public void clearKeyframes(V staticValue) {
		_staticValue = staticValue;
		_keyframes.clear();
	}

	public String getExpression() {
		return _expression;
	}

	public void setExpression(String expression) {
		_expression = expression;
	}

	protected V valueAtTime(double t, RenderContext renderContext) {
		int timeScale = renderContext.getTime().timeScale;
		Time time = new Time((long) (t * timeScale), timeScale); 
		return valueAtTime(time, renderContext);
	}

}
