/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Collections;
import java.util.List;

import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.AbstractTransformableLayer;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.AnimatableVec2d;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.BlendMode;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.CollapseTransformation;
import ch.kuramo.javie.core.DepthBuffer;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.RenderContext;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VectorMediaInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.VideoLayerBuffer;
import ch.kuramo.javie.core.VideoLayerComposer;
import ch.kuramo.javie.core.VideoLayerRenderer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.AudioEffectPipeline;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.VideoEffectPipeline;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public abstract class AbstractMediaLayer extends AbstractTransformableLayer implements MediaLayer {

	private boolean _videoNature;

	private boolean _audioNature;

	private boolean _videoEnabled;

	private boolean _audioEnabled;

	private boolean _effectsEnabled;

	private List<Effect> _effects;

	private AnimatableVec2d _depthBase;

	private AnimatableString _intersectionGroup;

	private AnimatableDouble _opacity;

	private BlendMode _blendMode;

	private AnimatableVec2d _audioLevels;

	private AnimatableDouble _timeRemap;

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private VideoEffectPipeline _vePipeline;

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private AudioEffectPipeline _aePipeline;


	@Override
	protected void initialize(String name) {
		throw new UnsupportedOperationException("Use initialize(String, boolean, boolean) method instead.");
	}

	protected void initialize(String name, boolean videoAvailable, boolean audioAvailable) {
		if (!videoAvailable && !audioAvailable) {
			throw new IllegalArgumentException();
		}

		super.initialize(name);

		if (videoAvailable) {
			_videoNature = true;
			_videoEnabled = true;
		}

		if (audioAvailable) {
			_audioNature = true;
			_audioEnabled = true;
		}

		_effectsEnabled = true;
		_effects = Util.newList();

		_depthBase = new AnimatableVec2d(new Vec2d(0, 0));
		_intersectionGroup = new AnimatableString("");
		_opacity = new AnimatableDouble(100d);
		_blendMode = BlendMode.NORMAL;
		_audioLevels = new AnimatableVec2d(new Vec2d(0, 0));
		_timeRemap = new AnimatableDouble(0.0);
	}

	public boolean isVideoNature() {
		return _videoNature;
	}

	public void setVideoNature(boolean videoNature) {
		_videoNature = videoNature;
	}

	public boolean isAudioNature() {
		return _audioNature;
	}

	public void setAudioNature(boolean audioNature) {
		_audioNature = audioNature;
	}

	public boolean isVideoEnabled() {
		return _videoEnabled;
	}

	public void setVideoEnabled(boolean enabled) {
		_videoEnabled = enabled;
	}

	public boolean isAudioEnabled() {
		return _audioEnabled;
	}

	public void setAudioEnabled(boolean enabled) {
		_audioEnabled = enabled;
	}

	public boolean isEffectsEnabled() {
		return _effectsEnabled;
	}

	public void setEffectsEnabled(boolean effectsEnabled) {
		_effectsEnabled = effectsEnabled;
	}

	public List<Effect> getEffects() {
		return _effects;
	}

	public void setEffects(List<Effect> effects) {
		_effects = effects;
	}

	public AnimatableVec2d getDepthBase() {
		return _depthBase;
	}

	public void setDepthBase(AnimatableVec2d depthBase) {
		_depthBase = depthBase;
	}

	public AnimatableString getIntersectionGroup() {
		return _intersectionGroup;
	}

	public void setIntersectionGroup(AnimatableString intersectionGroup) {
		_intersectionGroup = intersectionGroup;
	}

	public AnimatableDouble getOpacity() {
		return _opacity;
	}

	public void setOpacity(AnimatableDouble opacity) {
		_opacity = opacity;
	}

	public BlendMode getBlendMode() {
		return _blendMode;
	}

	public void setBlendMode(BlendMode blendMode) {
		_blendMode = blendMode;
	}

	public AnimatableVec2d getAudioLevels() {
		return _audioLevels;
	}

	public void setAudioLevels(AnimatableVec2d audioLevels) {
		_audioLevels = audioLevels;
	}

	public AnimatableDouble getTimeRemap() {
		return _timeRemap;
	}

	public void setTimeRemap(AnimatableDouble timeRemap) {
		_timeRemap = timeRemap;
	}

	public boolean isVectorLayer() {
		return (getMediaInput() instanceof VectorMediaInput);
	}

	@Override
	public void prepareExpression(ExpressionScope scope) {
		super.prepareExpression(scope);

		scope.assignTo(_depthBase);
		scope.assignTo(_intersectionGroup);
		scope.assignTo(_opacity);
		scope.assignTo(_audioLevels);
		scope.assignTo(_timeRemap);

		for (Effect effect : _effects) {
			effect.prepareExpression(scope.clone());
		}
	}

	public void setupVideoRenderer(
			VideoLayerComposer composer, Camera camera, CollapseTransformation ct) {

		if (!LayerNature.isVideoEnabled(this)) {
			return;
		}

		MediaInput input = getMediaInput();
		if (input == null || !input.isVideoAvailable()) {
			return;
		}

		if (isVectorLayer() && LayerNature.isCTCR(AbstractMediaLayer.this)) {
			setupVideoRendererCR(composer, camera, ct);
		} else {
			setupVideoRendererNormal(composer, camera, ct);
		}
	}

	private void setMediaTime(RenderContext context, MediaInput input) {
		Time time;

		if (LayerNature.isTimeRemapEnabled(this)) {
			if (context == _vrContext) {
				Time frameDuration = input.getVideoFrameDuration();
				int timeScale = frameDuration.timeScale;

				double t0 = _timeRemap.value(context);
				double t1 = _timeRemap.valueAtTime(context.getTime().add(frameDuration), context);
																	// TODO ↑ t1はコンポジションのframeDurationで計算すべきかも。

				time = new Time(Math.round(t0 * timeScale), timeScale);

				if (t0 > t1) {
					time = time.subtract(frameDuration);
				}

			} else if (context == _arContext) {
				int timeScale = _arContext.getAudioMode().sampleRate;
				time = new Time(Math.round(_timeRemap.value(context) * timeScale), timeScale);

				// TODO 変化量が負の時、オーディオは MediaInput のサンプルレートにおける１フレーム分をずらす必要がある。
				//      しかし、再生速度を変更している場合のオーディオの処理は今のところデタラメなので保留。

			} else {
				throw new IllegalArgumentException();
			}

		} else {
			time = context.getTime().subtract(getStartTime());

			double rate = getRate();
			if (rate != 1.0) {
				time = new Time(Math.round(time.timeValue * rate), time.timeScale);

				if (rate < 0.0) {
					time = time.add(input.getDuration());

					// 逆再生時は１フレーム分ずらす
					if (context == _vrContext) {
						Time frameDuration = input.getVideoFrameDuration();
						time = time.subtract(frameDuration);

					} else if (context == _arContext) {
						// TODO オーディオの場合、MediaInput のサンプルレートにおける１フレーム分をずらす必要がある。
						//      しかし、再生速度を変更している場合のオーディオの処理は今のところデタラメなので保留。

					} else {
						throw new IllegalArgumentException();
					}
				}
			}
		}

		context.setTime(time);
	}

	private void setupVideoRendererNormal(
			VideoLayerComposer composer, final Camera camera, final CollapseTransformation ct) {

		final MediaInput input = getMediaInput();
		RenderResolution resolution = _vrContext.getRenderResolution();

		VideoBounds bounds = _vePipeline.getVideoBounds(
				_effectsEnabled ? _effects : Collections.<Effect>emptyList(), resolution.scale(input.getVideoFrameBounds()));
		Vec2d offset = new Vec2d(-bounds.x/resolution.scale, -bounds.y/resolution.scale);

		final double[] mvMatrix = new double[16];
		final double[] prjMatrix = new double[16];
		createMatrix(camera, ct, offset, mvMatrix, prjMatrix);

		final double opacity;
		if (ct == null) {
			opacity = _opacity.value(_vrContext);
		} else {
			opacity = _opacity.value(_vrContext) * ct.getOpacity() / 100d;
		}

		VideoLayerRenderer r = new VideoLayerRenderer() {
			public VideoLayerBuffer render(boolean withDepthBuffer) {
				VideoBuffer effectedBuffer = null;
				try {
					VideoBuffer inputBuffer = _vrContext.saveAndExecute(new WrappedOperation<VideoBuffer>() {
						public VideoBuffer execute() {
							setMediaTime(_vrContext, input);
							return input.getVideoFrameImage();
						}
					});

					effectedBuffer = _vePipeline.doVideoEffects(
							_effectsEnabled ? _effects : Collections.<Effect>emptyList(), inputBuffer);

					VideoBuffer trnsfrmdBuffer = _vrSupport.createVideoBuffer(_vrContext.getColorMode(), camera.getViewportSize());
					trnsfrmdBuffer.allocateAsTexture();
					trnsfrmdBuffer.clear();

					DepthBuffer depthBuffer = withDepthBuffer ? _vrSupport.createDepthBuffer(camera.getViewportSize()) : null;

					_vrSupport.transform(effectedBuffer, trnsfrmdBuffer, depthBuffer, mvMatrix, prjMatrix, opacity);

					// TODO 親コンポがコラップスしている場合、プリコンポレイヤーのエフェクトはここ？

					return new VideoLayerBufferImpl(trnsfrmdBuffer, depthBuffer, _blendMode);

				} finally {
					if (effectedBuffer != null) {
						effectedBuffer.dispose();
					}
				}
			}
		};

		if (LayerNature.isThreeD(this)) {
			Vec2d depthBase2d = _depthBase.value(_vrContext);
			Vec3d depthBase = new Vec3d(depthBase2d.x + offset.x, depthBase2d.y + offset.y);
			depthBase = resolution.scale(depthBase);

			Size2i vp = camera.getViewportSize();
			Vec3d[] vertices = new Vec3d[] {
				_vrSupport.project(depthBase, mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y + bounds.height), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y + bounds.height), mvMatrix, prjMatrix, vp)
			};
			composer.add3D(r, vertices, _intersectionGroup.value(_vrContext));

		} else {
			composer.add2D(r);
		}
	}

	private void setupVideoRendererCR(
			VideoLayerComposer composer, final Camera camera, final CollapseTransformation ct) {

		final VectorMediaInput input = (VectorMediaInput) getMediaInput();
		RenderResolution resolution = _vrContext.getRenderResolution();

		VideoBounds bounds = resolution.scale(input.getVideoFrameBounds());

		final double[] mvMatrix = new double[16];
		final double[] prjMatrix = new double[16];
		createMatrix(camera, ct, null, mvMatrix, prjMatrix);

		final double opacity;
		if (ct == null) {
			opacity = _opacity.value(_vrContext);
		} else {
			opacity = _opacity.value(_vrContext) * ct.getOpacity() / 100d;
		}

		VideoLayerRenderer r = new VideoLayerRenderer() {
			public VideoLayerBuffer render(boolean withDepthBuffer) {
				final VideoBuffer rasterBuffer = _vrSupport.createVideoBuffer(_vrContext.getColorMode(), camera.getViewportSize());
				rasterBuffer.allocateAsTexture();
				rasterBuffer.clear();

				// TODO 連続ラスタライズの場合、デプスバッファはここで作成せずに全てのエフェクトが終わった後で計算する。
				final DepthBuffer depthBuffer = withDepthBuffer ? _vrSupport.createDepthBuffer(camera.getViewportSize()) : null;

				_vrContext.saveAndExecute(new WrappedOperation<Object>() {
					public Object execute() {
						setMediaTime(_vrContext, input);
						input.rasterize(rasterBuffer, depthBuffer, mvMatrix, prjMatrix, opacity);
						return null;
					}
				});

				// TODO このレイヤーのエフェクト
				// TODO 親コンポがコラップスしている場合、プリコンポレイヤーのエフェクトはここ？

				return new VideoLayerBufferImpl(rasterBuffer, depthBuffer, _blendMode);
			}
		};

		if (LayerNature.isThreeD(this)) {
			Vec2d depthBase2d = _depthBase.value(_vrContext);
			Vec3d depthBase = new Vec3d(depthBase2d.x, depthBase2d.y);
			depthBase = resolution.scale(depthBase);

			Size2i vp = camera.getViewportSize();
			Vec3d[] vertices = new Vec3d[] {
				_vrSupport.project(depthBase, mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y + bounds.height), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y + bounds.height), mvMatrix, prjMatrix, vp)
			};
			composer.add3D(r, vertices, _intersectionGroup.value(_vrContext));

		} else {
			composer.add2D(r);
		}
	}

	private void createMatrix(
			final Camera camera, final CollapseTransformation ct, final Vec2d offset,
			final double[] mvMatrix, final double[] prjMatrix) {

		WrappedOperation<Object> operation = new WrappedOperation<Object>() {
			public Object execute() {
				if (camera == null) {
					_vrSupport.resetMatrix();
				} else if (LayerNature.isThreeD(AbstractMediaLayer.this)) {
					_vrSupport.setMatrix(camera.getModelView3D(), camera.getProjection3D());
				} else {
					_vrSupport.setMatrix(camera.getModelView2D(), camera.getProjection2D());
				}

				LayerMatrixUtil.multModelViewMatrix(AbstractMediaLayer.this, offset, ct, _vrContext, _vrSupport);

				_vrSupport.getMatrix(mvMatrix, prjMatrix);
				return null;
			}
		};

		_vrSupport.pushMatrixAndExecute(operation);
	}

	public AudioBuffer renderAudioChunk() {
		if (!LayerNature.isAudioEnabled(this)) {
			return null;
		}

		final MediaInput input = getMediaInput();
		if (input == null || !input.isAudioAvailable()) {
			return null;
		}

		AudioBuffer ab = _arContext.saveAndExecute(new WrappedOperation<AudioBuffer>() {
			public AudioBuffer execute() {
				setMediaTime(_arContext, input);
				return input.getAudioChunk();
			}
		});

		if (ab == null) {
			return null;
		}

		ab = _aePipeline.doAudioEffects(
				_effectsEnabled ? _effects : Collections.<Effect>emptyList(), ab);

		// TODO オーディオレベルをアニメートしている場合、サンプル毎にオーディオレベルの値は異なるため、
		//      バッファ全体を同じオーディオレベルで処理するのは正しくない。
		Vec2d audioLevels = _audioLevels.value(_arContext);
		if (audioLevels.x != 0 || audioLevels.y != 0) {
			_arSupport.amplify(ab, audioLevels);
		}

		return ab;
	}

	public void afterDecode(Project p, LayerComposition c) throws ProjectDecodeException {
		super.afterDecode(p, c);

		for (Effect effect : _effects) {
			effect.afterDecode(p);
		}
	}

	public Object createExpressionElement(RenderContext renderContext) {
		return new MediaLayerExpressionElement(renderContext);
	}

	public class MediaLayerExpressionElement extends TransformableLayerExpressionElement {

		public MediaLayerExpressionElement(RenderContext renderContext) {
			super(renderContext);
		}

		public Object getDepthBase()			{ return elem(_depthBase); }
		public Object getIntersectionGroup()	{ return elem(_intersectionGroup); }
		public Object getOpacity()				{ return elem(_opacity); }
		public Object getAudioLevels()			{ return elem(_audioLevels); }
		public Object getTimeRemap()			{ return LayerNature.isTimeRemapEnabled(AbstractMediaLayer.this) ? elem(_timeRemap) : null; }

		public Object effect(int index) {
			Effect effect = _effects.get(index - 1);
			return renderContext.getExpressionElement(effect);
		}
	}

}
