/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.core.DepthBuffer;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public class DepthBufferImpl implements DepthBuffer {

	private static final Logger _logger = LoggerFactory.getLogger(DepthBufferImpl.class);

	@Inject
	private VideoRenderSupport _vrSupport;

	private final Size2i _imageSize;

	private int _texture;


	public DepthBufferImpl(Size2i imageSize) {
		_imageSize = imageSize;
	}

	@Override
	protected void finalize() throws Throwable {
		if (_texture != 0) {
			// TODO 解放忘れの場合どうする？
			//		finalize は別のスレッドから呼ばれるので、ここでテクスチャの解放をすることはできない。
			//		テクスチャ生成時に VideoRenderContext に登録しておいて、
			//		VideoRenderContext の deactivate で検出／解放するといいかも。
			_logger.warn("finalizing a DepthBufferImpl object, but the object is not disposed.");
		}

		super.finalize();
	}

	public void dispose() {
		if (_texture != 0) {
			_vrSupport.deleteTexture(_texture);
			_texture = 0;
		}
	}

	public int getTexture() {
		if (_texture == 0) {
			_texture = _vrSupport.createDepthTexture(_imageSize);
		}
		return _texture;
	}

	public Size2i getImageSize() {
		return _imageSize;
	}

}
